<?php
require_once __DIR__ . '/../config.php';

/**
 * =========================================================
 *  File: api_comments.php
 *  Compatível com o código "antigo" que você enviou
 *  Funcionalidades:
 *   - GET  ?feed_id=ID&sort=new|top[&limit=N&offset=M][&only=count]
 *        -> { success, comments:[...], count }
 *   - POST JSON {feed_id, comment_text, parent_comment_id?, poll?}
 *        -> { success, comment_id, new_count, comment }
 *   - PUT  JSON {comment_id, comment_text}
 *        -> { success }
 *   - DELETE ?comment_id=ID
 *        -> { success, new_count }
 *  Observações:
 *   - Mantém tabelas: comments, comments_polls, comments_poll_options, comments_poll_votes
 *   - JOIN com user_myhashtag (ID -> user_id) exatamente como seu antigo
 *   - Não exige refresh: POST/DELETE já devolvem contagem atualizada
 * =========================================================
 */
session_start();

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

$host    = 'localhost';
$db      = 'digupdog_FEED';
$user    = 'digupdog_FEEDadmin';
$pass    = 'Raimundinho1';
$charset = 'utf8mb4';

/* ======================== CONEXÃO DB ======================== */
try {
    $pdo = new PDO(
        "mysql:host={$host};dbname={$db};charset={$charset}",
        $user,
        $pass,
        [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]
    );
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['success'=>false,'error'=>'db_connection_failed']);
    exit;
}

/* ======================== HELPERS ======================== */
function table_exists(PDO $pdo, string $table): bool {
    static $cache = [];
    if (array_key_exists($table, $cache)) return $cache[$table];
    $st = $pdo->prepare("SELECT 1 FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? LIMIT 1");
    $st->execute([$table]);
    return $cache[$table] = (bool)$st->fetchColumn();
}

function currentUser(PDO $pdo): ?array {
    if (empty($_SESSION['username'])) return null;
    $st = $pdo->prepare("SELECT ID, username, user_role, profile_picture FROM user_myhashtag WHERE username = :u LIMIT 1");
    $st->execute([':u' => $_SESSION['username']]);
    $r = $st->fetch();
    return $r ?: null;
}
function isAdmin(?array $me): bool {
    return $me && isset($me['user_role']) && strtolower((string)$me['user_role']) === 'admin';
}

function extractFirstUrl(string $text): ?string {
    if (preg_match('~https?://[^\s]+~i', $text, $m)) return $m[0];
    return null;
}
function absolutizeUrl(string $base, string $rel): string {
    if ($rel === '') return $rel;
    if (preg_match('~^https?://~i', $rel)) return $rel;
    $p = parse_url($base);
    if (!$p) return $rel;
    $scheme = $p['scheme'] ?? 'http';
    $host   = $p['host'] ?? '';
    $port   = isset($p['port']) ? ':'.$p['port'] : '';
    $path   = $p['path'] ?? '/';
    $path   = preg_replace('~/[^/]*$~', '/', $path);
    if ($rel[0] === '/') $path = '';
    return $scheme.'://'.$host.$port.$path.ltrim($rel,'/');
}
function fetchOpenGraph(string $url, int $timeoutMs = 3500): array {
    $ua = 'Mozilla/5.0 (compatible; MyHashtagBot/1.0; +https://myhashtag.io)';
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER     => true,
        CURLOPT_FOLLOWLOCATION     => true,
        CURLOPT_MAXREDIRS          => 5,
        CURLOPT_CONNECTTIMEOUT_MS  => $timeoutMs,
        CURLOPT_TIMEOUT_MS         => $timeoutMs + 1000,
        CURLOPT_USERAGENT          => $ua,
        CURLOPT_ENCODING           => 'gzip,deflate',
        CURLOPT_SSL_VERIFYPEER     => false,
        CURLOPT_SSL_VERIFYHOST     => false,
        CURLOPT_HEADER             => false,
    ]);
    $html = curl_exec($ch);
    $finalUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL) ?: $url;
    curl_close($ch);

    if (!$html) return ['title'=>null,'description'=>null,'image'=>null,'final'=>$finalUrl];

    $title = null;
    if (preg_match('~<title[^>]*>(.*?)</title>~is', $html, $m)) {
        $title = trim(html_entity_decode($m[1], ENT_QUOTES|ENT_HTML5, 'UTF-8'));
    }

    $meta = [];
    if (preg_match_all('~<meta[^>]+(?:property|name)=["\']([^"\']+)["\'][^>]*content=["\']([^"\']+)["\'][^>]*>~i', $html, $ms, PREG_SET_ORDER)) {
        foreach ($ms as $mm) {
            $meta[strtolower($mm[1])] = html_entity_decode($mm[2], ENT_QUOTES|ENT_HTML5, 'UTF-8');
        }
    }
    $ogTitle = $meta['og:title'] ?? $meta['twitter:title'] ?? $title;
    $ogDesc  = $meta['og:description'] ?? $meta['twitter:description'] ?? null;
    $ogImg   = $meta['og:image'] ?? $meta['twitter:image'] ?? null;
    if ($ogImg) $ogImg = absolutizeUrl($finalUrl, $ogImg);

    return ['title'=>$ogTitle, 'description'=>$ogDesc, 'image'=>$ogImg, 'final'=>$finalUrl];
}
function countComments(PDO $pdo, int $feedId): int {
    $st = $pdo->prepare("SELECT COUNT(*) FROM comments WHERE feed_id = ?");
    $st->execute([$feedId]);
    return (int)$st->fetchColumn();
}

/* ======================== GET ======================== */
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $feedId = isset($_GET['feed_id']) ? (int)$_GET['feed_id'] : 0;
    if ($feedId <= 0) { echo json_encode(['success'=>false,'error'=>'invalid_feed','comments'=>[]]); exit; }

    // atalho: only=count (rápido)
    if (isset($_GET['only']) && $_GET['only'] === 'count') {
        echo json_encode(['success'=>true, 'comments'=>[], 'count'=>countComments($pdo, $feedId)]); exit;
    }

    $sort    = isset($_GET['sort']) ? $_GET['sort'] : 'new';
    $limit   = isset($_GET['limit'])  ? max(0, min(1000, (int)$_GET['limit']))  : 0;
    $offset  = isset($_GET['offset']) ? max(0, (int)$_GET['offset'])            : 0;

    // Ordenação compatível com seu antigo
    $sortSql = "c.created_at ASC, c.comment_id ASC";
    if ($sort === 'top') {
        // "Top" ingênuo: mais replies primeiro, depois por tempo (exatamente como seu antigo)
        $sortSql = " (SELECT COUNT(*) FROM comments cc WHERE cc.parent_comment_id = c.comment_id) DESC, c.created_at ASC";
    }

    try {
        $me   = currentUser($pdo);
        $meId = (int)($me['ID'] ?? 0);

        $sql = "SELECT
                    c.comment_id, c.feed_id, c.user_id, c.comment_text, c.created_at, c.parent_comment_id,
                    c.link, c.title, c.description, c.thumbnail,
                    u.username, u.profile_picture
                FROM comments c
                LEFT JOIN user_myhashtag u ON u.ID = c.user_id
                WHERE c.feed_id = :fid
                ORDER BY {$sortSql}";
        if ($limit > 0) $sql .= " LIMIT {$limit} OFFSET {$offset}";

        $st = $pdo->prepare($sql);
        $st->execute([':fid'=>$feedId]);
        $rows = $st->fetchAll();

        // Índice por comment_id para anexar polls
        $byCid = [];
        $cids  = [];
        foreach ($rows as &$r) { $byCid[(int)$r['comment_id']] = &$r; $cids[] = (int)$r['comment_id']; }

        // Polls (só se tabelas existirem)
        $hasPolls = table_exists($pdo, 'comments_polls')
                 && table_exists($pdo, 'comments_poll_options')
                 && table_exists($pdo, 'comments_poll_votes');

        if ($hasPolls && $cids) {
            $in  = implode(',', array_fill(0, count($cids), '?'));
            $ps  = $pdo->prepare("SELECT p.poll_id, p.comment_id, p.question FROM comments_polls p WHERE p.comment_id IN ($in)");
            $ps->execute($cids);
            $polls = $ps->fetchAll();

            if ($polls) {
                $pids = array_map(fn($p)=>(int)$p['poll_id'], $polls);
                $inP  = implode(',', array_fill(0, count($pids), '?'));

                $os = $pdo->prepare("SELECT o.option_id, o.poll_id, o.option_text FROM comments_poll_options o WHERE o.poll_id IN ($inP)");
                $os->execute($pids);
                $opts = $os->fetchAll();

                $vs = $pdo->prepare("SELECT v.poll_id, v.option_id, v.user_id FROM comments_poll_votes v WHERE v.poll_id IN ($inP)");
                $vs->execute($pids);
                $votes = $vs->fetchAll();

                $optByPoll   = [];
                foreach ($opts as $o) { $optByPoll[(int)$o['poll_id']][] = $o; }

                $votesByPoll = [];
                $mineByPoll  = [];
                foreach ($votes as $v) {
                    $pid = (int)$v['poll_id'];
                    $votesByPoll[$pid][] = $v;
                    if ($meId && (int)$v['user_id'] === $meId) $mineByPoll[$pid] = (int)$v['option_id'];
                }

                foreach ($polls as $p) {
                    $pid = (int)$p['poll_id'];
                    $osL = $optByPoll[$pid] ?? [];
                    $vsL = $votesByPoll[$pid] ?? [];
                    $total = count($vsL);
                    $mine  = $mineByPoll[$pid] ?? 0;

                    $optOut = [];
                    foreach ($osL as $o) {
                        $oid = (int)$o['option_id'];
                        $cnt = 0;
                        foreach ($vsL as $vv) if ((int)$vv['option_id'] === $oid) $cnt++;
                        $optOut[] = [
                            'option_id'   => $oid,
                            'option_text' => $o['option_text'],
                            'votes'       => $cnt,
                            'mine'        => ($mine && $oid === $mine)
                        ];
                    }

                    $payload = [
                        'poll_id'     => $pid,
                        'question'    => $p['question'],
                        'total_votes' => $total,
                        'options'     => $optOut
                    ];

                    $cid = (int)$p['comment_id'];
                    if (isset($byCid[$cid])) $byCid[$cid]['poll'] = $payload;
                }
            }
        }

        // Permissões (como seu antigo)
        foreach ($rows as &$r) {
            $r['can_edit']   = ($meId && (int)$r['user_id'] === $meId) || isAdmin($me);
            $r['can_delete'] = ($meId && (int)$r['user_id'] === $meId) || isAdmin($me);
        }

        echo json_encode([
            'success' => true,
            'comments'=> $rows,
            'count'   => count($rows) // rápido; se quiser o total sem LIMIT, use countComments()
        ]);
        exit;

    } catch (Throwable $e) {
        echo json_encode(['success'=>false, 'error'=>'get_failed']); exit;
    }
}

/* ======================== POST ======================== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $me = currentUser($pdo);
    if (!$me) { echo json_encode(['success'=>false,'error'=>'not_logged_in']); exit; }

    $payload = json_decode(file_get_contents('php://input'), true) ?: [];
    $feedId  = isset($payload['feed_id']) ? (int)$payload['feed_id'] : 0;
    $text    = isset($payload['comment_text']) ? trim((string)$payload['comment_text']) : '';
    $parent  = isset($payload['parent_comment_id']) ? (int)$payload['parent_comment_id'] : null;
    $poll    = (isset($payload['poll']) && is_array($payload['poll'])) ? $payload['poll'] : null;

    if ($feedId <= 0 || $text === '') {
        echo json_encode(['success'=>false,'error'=>'invalid_params']); exit;
    }

    try {
        // URL preview (OG)
        $link = extractFirstUrl($text);
        $title = $desc = $thumb = null;
        if ($link) {
            $og    = fetchOpenGraph($link, 3500);
            $title = $og['title'] ?? null;
            $desc  = $og['description'] ?? null;
            $thumb = $og['image'] ?? null;
        }

        // Insert
        $st = $pdo->prepare(
            "INSERT INTO comments (feed_id, user_id, comment_text, parent_comment_id, link, title, description, thumbnail, created_at)
             VALUES (:fid, :uid, :txt, :pid, :lnk, :ttl, :des, :thb, NOW())"
        );
        $st->execute([
            ':fid' => $feedId,
            ':uid' => (int)$me['ID'],
            ':txt' => $text,
            ':pid' => $parent,
            ':lnk' => $link,
            ':ttl' => $title,
            ':des' => $desc,
            ':thb' => $thumb
        ]);
        $cid = (int)$pdo->lastInsertId();

        // Poll opcional (se existir schema)
        if ($poll
            && table_exists($pdo, 'comments_polls')
            && table_exists($pdo, 'comments_poll_options')
        ) {
            if (!empty($poll['question']) && !empty($poll['options']) && is_array($poll['options']) && count($poll['options']) >= 2) {
                $pdo->prepare("INSERT INTO comments_polls (comment_id, question, created_at) VALUES (?, ?, NOW())")
                    ->execute([$cid, trim((string)$poll['question'])]);
                $pid = (int)$pdo->lastInsertId();
                $insOpt = $pdo->prepare("INSERT INTO comments_poll_options (poll_id, option_text) VALUES (?, ?)");
                foreach ($poll['options'] as $opt) {
                    $opt = trim((string)$opt);
                    if ($opt !== '') $insOpt->execute([$pid, $opt]);
                }
            }
        }

        // Montar payload do comentário recém-criado (para UI não precisar refazer GET completo)
        $commentPayload = [
            'comment_id'        => $cid,
            'feed_id'           => $feedId,
            'user_id'           => (int)$me['ID'],
            'comment_text'      => $text,
            'created_at'        => date('Y-m-d H:i:s'),
            'parent_comment_id' => $parent,
            'link'              => $link,
            'title'             => $title,
            'description'       => $desc,
            'thumbnail'         => $thumb,
            'username'          => $me['username'],
            'profile_picture'   => $me['profile_picture'] ?? null,
            'can_edit'          => true,
            'can_delete'        => true
        ];

        $newCount = countComments($pdo, $feedId);

        echo json_encode(['success'=>true, 'comment_id'=>$cid, 'new_count'=>$newCount, 'comment'=>$commentPayload]);
        exit;

    } catch (Throwable $e) {
        echo json_encode(['success'=>false, 'error'=>'post_failed']); exit;
    }
}

/* ======================== PUT ======================== */
if ($_SERVER['REQUEST_METHOD'] === 'PUT') {
    $me = currentUser($pdo);
    if (!$me) { echo json_encode(['success'=>false,'error'=>'not_logged_in']); exit; }

    $payload = json_decode(file_get_contents('php://input'), true) ?: [];
    $cid  = isset($payload['comment_id']) ? (int)$payload['comment_id'] : 0;
    $text = isset($payload['comment_text']) ? trim((string)$payload['comment_text']) : '';

    if ($cid <= 0 || $text === '') { echo json_encode(['success'=>false,'error'=>'invalid_params']); exit; }

    try {
        // Ownership
        $st = $pdo->prepare("SELECT user_id FROM comments WHERE comment_id = ?");
        $st->execute([$cid]);
        $uid = (int)($st->fetchColumn() ?: 0);
        if (!$uid) { echo json_encode(['success'=>false,'error'=>'not_found']); exit; }
        if (!isAdmin($me) && $uid !== (int)$me['ID']) { echo json_encode(['success'=>false,'error'=>'forbidden']); exit; }

        $up = $pdo->prepare("UPDATE comments SET comment_text = :t WHERE comment_id = :id");
        $up->execute([':t' => $text, ':id' => $cid]);

        echo json_encode(['success'=>true]); exit;

    } catch (Throwable $e) {
        echo json_encode(['success'=>false,'error'=>'put_failed']); exit;
    }
}

/* ======================== DELETE ======================== */
if ($_SERVER['REQUEST_METHOD'] === 'DELETE') {
    $me = currentUser($pdo);
    if (!$me) { echo json_encode(['success'=>false,'error'=>'not_logged_in']); exit; }

    $cid = isset($_GET['comment_id']) ? (int)$_GET['comment_id'] : 0;
    if ($cid <= 0) { echo json_encode(['success'=>false,'error'=>'invalid_params']); exit; }

    try {
        // Descobrir feed_id e ownership
        $st = $pdo->prepare("SELECT user_id, feed_id FROM comments WHERE comment_id = ? LIMIT 1");
        $st->execute([$cid]);
        $row = $st->fetch();
        if (!$row) { echo json_encode(['success'=>false,'error'=>'not_found']); exit; }

        $uid    = (int)$row['user_id'];
        $feedId = (int)$row['feed_id'];

        if (!isAdmin($me) && $uid !== (int)$me['ID']) { echo json_encode(['success'=>false,'error'=>'forbidden']); exit; }

        // Remover poll vinculado (se existir schema)
        if (table_exists($pdo, 'comments_polls')) {
            $st = $pdo->prepare("SELECT poll_id FROM comments_polls WHERE comment_id = ? LIMIT 1");
            $st->execute([$cid]);
            $pid = (int)($st->fetchColumn() ?: 0);
            if ($pid) {
                if (table_exists($pdo, 'comments_poll_votes')) {
                    $pdo->prepare("DELETE FROM comments_poll_votes   WHERE poll_id = ?")->execute([$pid]);
                }
                if (table_exists($pdo, 'comments_poll_options')) {
                    $pdo->prepare("DELETE FROM comments_poll_options WHERE poll_id = ?")->execute([$pid]);
                }
                $pdo->prepare("DELETE FROM comments_polls       WHERE poll_id = ?")->execute([$pid]);
            }
        }

        // Remover comentário (mantém o comportamento do seu antigo — não apaga replies em cascata)
        $pdo->prepare("DELETE FROM comments WHERE comment_id = ?")->execute([$cid]);

        $newCount = countComments($pdo, $feedId);
        echo json_encode(['success'=>true, 'new_count'=>$newCount]); exit;

    } catch (Throwable $e) {
        echo json_encode(['success'=>false,'error'=>'delete_failed']); exit;
    }
}

/* ======================== Fallback ======================== */
http_response_code(405);
echo json_encode(['success'=>false,'error'=>'method_not_allowed']);
