<?php
require_once __DIR__ . '/../config.php';

/**
 * =========================================================
 *  File: api_share.php
 *  Purpose:
 *    - GET  ?feed_id=ID            -> { success:true, count:int }
 *    - POST { feed_id, channel? }  -> increment and return { success:true, count:int }
 *  Notes:
 *    - Keeps existing `shares` table.
 *    - When POSTing, also upserts into `share_stats` if channel is provided.
 *    - Optional lightweight IP rate‑limit (per feed_id, 2s).
 * =========================================================
 */
header('Content-Type: application/json; charset=utf-8');

$host    = 'localhost';
$db      = 'digupdog_FEED';
$user    = 'digupdog_FEEDadmin';
$pass    = 'Raimundinho1';
$charset = 'utf8mb4';

function client_ip(): string {
    return $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
}

try {
    $pdo = new PDO(
        "mysql:host={$host};dbname={$db};charset={$charset}",
        $user,
        $pass,
        [ PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
          PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC ]
    );
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'db_connection_failed']);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    $fid = isset($_GET['feed_id']) ? (int)$_GET['feed_id'] : 0;
    if ($fid <= 0) {
        echo json_encode(['success' => false, 'error' => 'invalid_feed_id']);
        exit;
    }
    $st  = $pdo->prepare("SELECT count FROM shares WHERE feed_id = ?");
    $st->execute([$fid]);
    $cnt = (int)($st->fetchColumn() ?: 0);
    echo json_encode(['success' => true, 'count' => $cnt]);
    exit;
}

if ($method === 'POST') {
    $payload = json_decode(file_get_contents('php://input'), true) ?: [];
    $fid     = (int)($payload['feed_id'] ?? 0);
    $channel = isset($payload['channel']) ? trim((string)$payload['channel']) : '';

    if ($fid <= 0) {
        echo json_encode(['success' => false, 'error' => 'invalid_feed_id']);
        exit;
    }

    // Simple per-IP rate limit: 1 share per feed_id every 2 seconds.
    // Uses feed_shares if available; if not, skip silently.
    try {
        $ip     = client_ip();
        $recent = $pdo->prepare("SELECT created_at FROM feed_shares WHERE feed_id = ? AND ip = ? ORDER BY created_at DESC LIMIT 1");
        $recent->execute([$fid, $ip]);
        $last = $recent->fetchColumn();
        if ($last) {
            $lastTs = strtotime($last);
            if ($lastTs !== false && (time() - $lastTs) < 2) {
                // Still increment `shares` but avoid spamming `feed_shares`.
                // This keeps counters responsive but lowers log spam.
                $pdo->prepare("INSERT INTO shares (feed_id, count, last_share_at)
                               VALUES (?, 1, NOW())
                               ON DUPLICATE KEY UPDATE count = count + 1, last_share_at = NOW()")
                    ->execute([$fid]);

                $st  = $pdo->prepare("SELECT count FROM shares WHERE feed_id = ?");
                $st->execute([$fid]);
                $cnt = (int)($st->fetchColumn() ?: 0);
                echo json_encode(['success' => true, 'count' => $cnt, 'rate_limited' => true]);
                exit;
            }
        }
    } catch (Throwable $e) {
        // no-op; do not break core flow
    }

    // Increment global counter
    $pdo->prepare(
        "INSERT INTO shares (feed_id, count, last_share_at)
         VALUES (?, 1, NOW())
         ON DUPLICATE KEY UPDATE count = count + 1, last_share_at = NOW()"
    )->execute([$fid]);

    // Optional per-channel stat
    if ($channel !== '') {
        $channel = substr(preg_replace('/[^a-z0-9\-\._]/i', '', $channel), 0, 32);
        if ($channel !== '') {
            $pdo->prepare(
                "INSERT INTO share_stats (feed_id, channel, count, last_shared_at)
                 VALUES (?, ?, 1, NOW())
                 ON DUPLICATE KEY UPDATE count = count + 1, last_shared_at = NOW()"
            )->execute([$fid, $channel]);
        }
    }

    // Log into feed_shares if table exists
    try {
        $ip = client_ip();
        $pdo->prepare(
            "INSERT INTO feed_shares (feed_id, user_id, channel, ip, created_at)
             VALUES (?, NULL, ?, ?, NOW())"
        )->execute([$fid, $channel ?: 'other', $ip]);
    } catch (Throwable $e) {
        // table might not exist or have different structure; ignore
    }

    $st  = $pdo->prepare("SELECT count FROM shares WHERE feed_id = ?");
    $st->execute([$fid]);
    $cnt = (int)($st->fetchColumn() ?: 0);

    echo json_encode(['success' => true, 'count' => $cnt]);
    exit;
}

http_response_code(405);
echo json_encode(['success' => false, 'error' => 'unsupported_method']);
