<?php
require_once __DIR__ . '/../config.php';

/**
 * =========================================================
 *  File: api_share_event.php
 *  Purpose: Log share event (feed_id, channel, user when available)
 *  Input: POST JSON { feed_id:int, channel:string }
 *  Output: { success: true }
 *  Notes:
 *    - Light IP rate limiting (2 seconds per feed/channel).
 *    - Also upserts share_stats and increments shares to keep numbers in sync.
 * =========================================================
 */
declare(strict_types=1);

session_start();
header('Content-Type: application/json; charset=utf-8');

$cfg = [
  'host'    => 'localhost',
  'db'      => 'digupdog_FEED',
  'user'    => 'digupdog_FEEDadmin',
  'pass'    => 'Raimundinho1',
  'charset' => 'utf8mb4'
];

function client_ip(): string {
    return $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
}

$raw = file_get_contents('php://input');
$js  = json_decode($raw, true) ?: [];
$feed_id = isset($js['feed_id']) ? (int)$js['feed_id'] : 0;
$channel = isset($js['channel']) ? trim((string)$js['channel']) : '';

if ($feed_id <= 0 || $channel === '') {
  echo json_encode(['success' => false, 'error' => 'bad_params']);
  exit;
}

$mysqli = @new mysqli($cfg['host'], $cfg['user'], $cfg['pass'], $cfg['db']);
if ($mysqli->connect_error) {
  echo json_encode(['success' => false, 'error' => 'db']);
  exit;
}
$mysqli->set_charset($cfg['charset']);

$user_id = null;
if (!empty($_SESSION['username'])) {
  $u     = $_SESSION['username'];
  $stmtU = $mysqli->prepare("SELECT id FROM user_myhashtag WHERE username = ? LIMIT 1");
  $stmtU->bind_param('s', $u);
  if ($stmtU->execute()) {
    $idrow = $stmtU->get_result()->fetch_assoc();
    if ($idrow) $user_id = (int)$idrow['id'];
  }
  $stmtU->close();
}

$ip = client_ip();
$channel = substr(preg_replace('/[^a-z0-9\-\._]/i', '', $channel), 0, 32);
if ($channel === '') $channel = 'other';

/** Simple IP rate‑limit: 1 event per feed/channel per 2 seconds for the same IP */
$rlOk = true;
if ($stmtR = $mysqli->prepare("SELECT created_at FROM feed_shares WHERE feed_id = ? AND channel = ? AND ip = ? ORDER BY created_at DESC LIMIT 1")) {
  $stmtR->bind_param('iss', $feed_id, $channel, $ip);
  if ($stmtR->execute()) {
    $last = $stmtR->get_result()->fetch_assoc();
    if ($last) {
      $lastTs = strtotime($last['created_at']);
      if ($lastTs !== false && (time() - $lastTs) < 2) $rlOk = false;
    }
  }
  $stmtR->close();
}

// Always increment global counter & per-channel counter (so the UI remains responsive)
// Log line item only when not rate-limited.
$ok = true;

try {
  // shares
  $stmt = $mysqli->prepare(
    "INSERT INTO shares (feed_id, count, last_share_at)
     VALUES (?, 1, NOW())
     ON DUPLICATE KEY UPDATE count = count + 1, last_share_at = NOW()"
  );
  $stmt->bind_param('i', $feed_id);
  $ok = $ok && $stmt->execute();
  $stmt->close();

  // share_stats
  $stmt2 = $mysqli->prepare(
    "INSERT INTO share_stats (feed_id, channel, count, last_shared_at)
     VALUES (?, ?, 1, NOW())
     ON DUPLICATE KEY UPDATE count = count + 1, last_shared_at = NOW()"
  );
  $stmt2->bind_param('iss', $feed_id, $channel, $ts = date('Y-m-d H:i:s')); // dummy bind for types
  // Re-bind properly without the $ts (we only need types as 'iss').
  $stmt2->bind_param('iss', $feed_id, $channel, $ch = $channel); // second param uses channel, third ignored
  // Simpler: re-prepare cleanly
  $stmt2->close();
  $stmt2 = $mysqli->prepare(
    "INSERT INTO share_stats (feed_id, channel, count, last_shared_at)
     VALUES (?, ?, 1, NOW())
     ON DUPLICATE KEY UPDATE count = count + 1, last_shared_at = NOW()"
  );
  $stmt2->bind_param('is', $feed_id, $channel);
  $ok = $ok && $stmt2->execute();
  $stmt2->close();

  if ($rlOk) {
    $stmt3 = $mysqli->prepare("INSERT INTO feed_shares (feed_id, user_id, channel, ip, created_at) VALUES (?, ?, ?, ?, NOW())");
    // If $user_id is null, bind_param will insert NULL correctly for mysqlnd
    $stmt3->bind_param('iiss', $feed_id, $user_id, $channel, $ip);
    $ok = $ok && $stmt3->execute();
    $stmt3->close();
  }
} catch (Throwable $e) {
  $ok = false;
}

echo json_encode(['success' => (bool)$ok], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
