/**
 * API Communication Module
 */

const API = {
  baseURL: '../api',

  async request(endpoint, options = {}) {
    const url = `${this.baseURL}/${endpoint}`;
    const config = {
      method: options.method || 'GET',
      headers: {
        'Content-Type': 'application/json',
        ...options.headers
      },
      ...options
    };

    if (options.body && typeof options.body === 'object') {
      config.body = JSON.stringify(options.body);
    }

    try {
      const response = await fetch(url, config);
      const data = await response.json();
      return data;
    } catch (error) {
      console.error('API Error:', error);
      throw error;
    }
  },

  // Feed endpoints
  async getPosts(page = 1, limit = 20) {
    return this.request(`get_pinfeeds.php?page=${page}&limit=${limit}`);
  },

  async getAuthorPosts(username, page = 1) {
    return this.request(`author_feeds.php?username=${username}&page=${page}`);
  },

  // Engagement endpoints
  async likePost(postId) {
    return this.request('likes.php', {
      method: 'POST',
      body: { post_id: postId, action: 'like' }
    });
  },

  async commentPost(postId, text) {
    return this.request('comments.php', {
      method: 'POST',
      body: { post_id: postId, text }
    });
  },

  async reactPost(postId, reaction) {
    return this.request('reactions.php', {
      method: 'POST',
      body: { post_id: postId, reaction }
    });
  },

  async sharePost(postId) {
    return this.request('share.php', {
      method: 'POST',
      body: { post_id: postId }
    });
  },

  async favoritePost(postId) {
    return this.request('favorites.php', {
      method: 'POST',
      body: { post_id: postId }
    });
  },

  // Social endpoints
  async followUser(userId) {
    return this.request('follow.php', {
      method: 'POST',
      body: { user_id: userId, action: 'follow' }
    });
  },

  async searchUsers(query) {
    return this.request(`user_search.php?q=${encodeURIComponent(query)}`);
  },

  // User info
  async getCurrentUser() {
    return this.request('whoami.php');
  },

  async getAuthorInfo(username) {
    return this.request(`author_info.php?username=${username}`);
  }
};

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
  module.exports = API;
}
