/**
 * DigUpDOG v2.0 - Main Application
 */

// Application state
const App = {
  currentUser: null,
  isInitialized: false,

  /**
   * Initialize application
   */
  async init() {
    if (this.isInitialized) return;

    console.log('🚀 Initializing DigUpDOG v2.0...');

    try {
      // Get current user
      this.currentUser = await API.getCurrentUser();

      // Initialize UI components
      this.initUI();

      // Load initial feed
      await this.loadInitialFeed();

      // Initialize tags
      if (typeof initializeTags === 'function') {
        initializeTags();
      }

      // Colorize navigation
      if (typeof colorizeNavItems === 'function') {
        setTimeout(() => colorizeNavItems(), 100);
      }

      this.isInitialized = true;
      console.log('✅ DigUpDOG initialized successfully');

    } catch (error) {
      console.error('❌ Initialization failed:', error);
    }
  },

  /**
   * Initialize UI event listeners
   */
  initUI() {
    // Mobile sidebar toggle
    const menuBtn = document.querySelector('.menu-btn');
    if (menuBtn) {
      menuBtn.addEventListener('click', () => UI.toggleSidebar());
    }

    // Modal close buttons
    document.querySelectorAll('.modal-close, .modal-backdrop').forEach(el => {
      el.addEventListener('click', (e) => {
        const modal = e.target.closest('.modal');
        if (modal) UI.closeModal(modal.id);
      });
    });

    // Close dropdowns on outside click
    document.addEventListener('click', (e) => {
      if (!e.target.closest('.dropdown')) {
        document.querySelectorAll('.dropdown.active').forEach(d => {
          d.classList.remove('active');
        });
      }
    });
  },

  /**
   * Load initial feed
   */
  async loadInitialFeed() {
    try {
      const response = await API.getPosts(1);

      if (response.success && response.posts) {
        await Feed.renderPosts(response.posts);
        Feed.hasMore = response.has_more;
      }
    } catch (error) {
      console.error('Failed to load feed:', error);
      UI.showToast('Failed to load feed', 'error');
    }
  },

  /**
   * Handle navigation
   */
  navigate(tab) {
    // Update active nav item
    document.querySelectorAll('.nav-item').forEach(item => {
      item.classList.remove('active');
    });

    const activeItem = document.querySelector(`.nav-item[data-tab="${tab}"]`);
    if (activeItem) {
      activeItem.classList.add('active');
    }

    // Load content based on tab
    this.loadTabContent(tab);
  },

  /**
   * Load content for specific tab
   */
  async loadTabContent(tab) {
    const container = document.querySelector('#feed-container');
    if (!container) return;

    container.innerHTML = '<div class="loading">Loading...</div>';

    // Reset feed state
    Feed.currentPage = 1;
    Feed.hasMore = true;

    // Load content based on tab
    await this.loadInitialFeed();
  }
};

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => App.init());
} else {
  App.init();
}

// Export for use in other modules
if (typeof module !== 'undefined' && module.exports) {
  module.exports = App;
}
