/*Feeds.js ================== STATE ================== */
var feeds = {};
var feedsById = {};
var loading = false;
var page = 1;
var loadThreshold = 20000;
var displayedFeeds = {};

var userFollowSets = { tags: new Set(), sourceDomains: new Set(), authors: new Set() };
var userFollowSetsLoaded = false;

var filterMode = 'all'; // 'all' | 'my'

/* ================== HELPERS (Colors/Toasts/Confetti/Modal) ================== */
function fetchUserFollowSetsOnce() {
  return new Promise(function(resolve){
    if (userFollowSetsLoaded) { resolve(); return; }
    $.getJSON('api_follow_sets.php', function(res){
      if (res && res.success) {
        (res.tags || []).forEach(t => userFollowSets.tags.add((t||'').toLowerCase()));
        (res.source_domains || []).forEach(d => userFollowSets.sourceDomains.add((d||'').toLowerCase()));
        (res.authors || []).forEach(a => userFollowSets.authors.add((a||'').toLowerCase()));
        userFollowSetsLoaded = true;
      }
      resolve();
    }).fail(function(){ resolve(); });
  });
}

function accColorFromText(txt){ try{ return stringToColor(txt || ''); }catch(e){ return '#4A90E2'; } }

function ensureToastOverlay(){
  var o = document.getElementById('toast-overlay');
  if (!o){ o = document.createElement('div'); o.id = 'toast-overlay'; document.body.appendChild(o); }
  return o;
}
function showToast({title, subtitle, icon='fa-check', color='#4A90E2'}){
  var overlay = ensureToastOverlay();
  var t = document.createElement('div');
  t.className = 'toast';
  t.style.setProperty('--accent', color);
  if(icon === 'fa-xmark') icon = 'fa-times';
  t.innerHTML = `
     <div class="head">
      <div class="icon"><i class="fas ${icon}"></i></div>
      <div>
        <div class="title">${title}</div>
        <div class="subtitle">${subtitle}</div>
      </div>
     </div>`;
  overlay.appendChild(t);
  setTimeout(function(){ t.classList.add('leave'); setTimeout(()=>t.remove(), 260); }, 2300);
}

function showFollowToast(type, value, followed){
  var prefix = type==='tag' ? '#' : (type==='author' ? '@' : '');
  var label  = `${prefix}${value}`;
  var color  = accColorFromText(value);
  showToast({
    title: followed ? 'Now following' : 'Unfollowed',
    subtitle: (type==='source_domain' ? 'Channel ' : (type==='author' ? 'Author ' : 'Tag ')) + label,
    icon: followed ? 'fa-check' : 'fa-times',
    color
  });
}
function showLikeToast(added, kind){
  showToast({
    title: added ? (kind==='like'?'Liked':'Disliked') : (kind==='like'?'Like removed':'Dislike removed'),
    subtitle: 'Thanks for your feedback',
    icon: kind==='like' ? (added?'fa-thumbs-up':'fa-undo') : (added?'fa-thumbs-down':'fa-undo'),
    color: kind==='like' ? '#ffca2b' : '#ff5c5c'
  });
}
function showSaveToast(added){
  showToast({
    title: added ? 'Saved to favorites' : 'Removed from favorites',
    subtitle: 'You can find it in your profile',
    icon: 'fa-bookmark',
    color: '#40c4ff'
  });
}
function showEmbedToast(){
  showToast({
    title: 'Embed code copied',
    subtitle: 'Paste it where you need the card',
    icon: 'fa-code',
    color: '#FFC107'
  });
}

/* Confetti (light) */
function fireConfetti(color){
  var layer = document.createElement('div');
  layer.className = 'confetti';
  var cx = (window.innerWidth/2) + 'px';
  var cy = (window.innerHeight*0.45) + 'px';
  for (var i=0;i<28;i++){
    var dot = document.createElement('i');
    dot.style.setProperty('--x', cx);
    dot.style.setProperty('--y', cy);
    var dx = (Math.random()*240 - 120).toFixed(0) + 'px';
    var dy = (Math.random()*-160 - 30).toFixed(0) + 'px';
    dot.style.setProperty('--dx', dx);
    dot.style.setProperty('--dy', dy);
    dot.style.setProperty('--c', color || '#fff');
    layer.appendChild(dot);
  }
  document.body.appendChild(layer);
  setTimeout(function(){ layer.remove(); }, 800);
}

/* Confirm Unfollow modal */
function ensureConfirmModal(){
  var overlay = document.getElementById('confirm-overlay');
  if (overlay) return overlay;
  overlay = document.createElement('div');
  overlay.id = 'confirm-overlay';
  overlay.innerHTML = `
    <div class="confirm-modal" role="dialog" aria-modal="true">
      <div class="cm-head">
        <div class="cm-icon"><i class="fas fa-user-times"></i></div>
        <h3 id="cm-title">Unfollow?</h3>
      </div>
      <p id="cm-sub">Stop seeing updates from this item.</p>
      <div class="cm-actions">
        <button class="btn cancel" id="cm-cancel">Cancel</button>
        <button class="btn danger" id="cm-confirm">Unfollow</button>
      </div>
    </div>`;
  document.body.appendChild(overlay);
  return overlay;
}
function confirmUnfollow(type, value, onConfirm){
  var overlay = ensureConfirmModal();
  var title = 'Unfollow?';
  var label = (type==='tag'?'#':type==='author'?'@':'') + value;
  var sub   = (type==='source_domain'?'Channel ':'') + label;
  overlay.querySelector('#cm-title').textContent = title;
  overlay.querySelector('#cm-sub').textContent = 'Stop seeing updates from ' + sub;
  overlay.style.display = 'flex';

  var cancel = overlay.querySelector('#cm-cancel');
  var confirm = overlay.querySelector('#cm-confirm');

  function close(){ overlay.style.display='none'; cancel.onclick = confirm.onclick = null; }

  cancel.onclick = close;
  confirm.onclick = function(){ close(); onConfirm && onConfirm(); };
}

/* ================== FILTERBAR (All / My Follows) ================== */
function ensureFilterBar(){
  if (document.getElementById('filterbar')) return;
  var bar = document.createElement('div');
  bar.id = 'filterbar';
  bar.className = 'feed-filterbar';
  bar.innerHTML = `
    <button class="filter-tab active" data-filter="all" title="Show all">
      <i class="fas fa-stream"></i> All
    </button>
    <button class="filter-tab" data-filter="my" title="Only authors/channels/tags you follow">
      <i class="fas fa-user-check"></i> My Follows
    </button>`;
  var anchor = document.getElementById('tags-container') || document.getElementById('feeds');
  anchor.parentNode.insertBefore(bar, anchor.nextSibling);

  bar.addEventListener('click', function(e){
    var btn = e.target.closest('.filter-tab');
    if (!btn) return;
    Array.from(bar.querySelectorAll('.filter-tab')).forEach(b=>b.classList.remove('active'));
    btn.classList.add('active');
    filterMode = btn.dataset.filter === 'my' ? 'my' : 'all';
    // Restart pagination + reload
    page = 1;
    if (window.masonry) window.masonry.clear(); else $('#feeds').empty();
    window.scrollTo({ top: 0, behavior: 'smooth' });
    loadFeeds();
  });
}

/* ================== In-memory follow filter (for 'all' mode) ================== */
function isFeedFollowed(feed){
  try{
    var author = (feed.author || '').toLowerCase();
    var domain = (extractBaseDomainFromUrl(feed.link) || feed.source_domain || '').toLowerCase();
    if (userFollowSets.authors.has(author)) return true;
    if (userFollowSets.sourceDomains.has(domain)) return true;
    var tags = extractTagsFromTitle(feed.title || '');
    for (var i=0;i<tags.length;i++){
      if (userFollowSets.tags.has((tags[i]||'').toLowerCase())) return true;
    }
  }catch(e){}
  return false;
}

/* ================== DISPLAY (when endpoint = get_pinfeeds.php) ================== */
function displayFeedsOnlyWithImage() {
  var container = $('#feeds');
  var filteredFeeds = [];
  $.each(feeds, function(key, feed) {
    var image = feed.image || '';
    var thumbnail = feed.thumbnail || '';
    if ((image || thumbnail) && !displayedFeeds[feed.link]) {
      if (filterMode === 'my' && !isFeedFollowed(feed)) return;
      displayedFeeds[feed.link] = true;
      filteredFeeds.push(feed);
    }
  });
  displayFeedsToContainer(filteredFeeds, container, true);
}
function displayFeedsOnlyWithImageAppend(shouldClear = false) {
  var container = $('#feeds');
  var filteredFeeds = [];
  $.each(feeds, function(key, feed) {
    var image = feed.image || '';
    var thumbnail = feed.thumbnail || '';
    if ((image || thumbnail) && !displayedFeeds[feed.link]) {
      if (filterMode === 'my' && !isFeedFollowed(feed)) return;
      displayedFeeds[feed.link] = true;
      filteredFeeds.push(feed);
    }
  });
  displayFeedsToContainer(filteredFeeds, container, shouldClear);
}
function displayFeeds() {
  var searchQuery = $('#search-query').val().toLowerCase();
  var filteredFeeds = [];

  if (searchQuery.startsWith("author:")) {
      var authorName = searchQuery.replace("author:", "").trim();
      filteredFeeds = $.map(feeds, function(feed) {
          if ((feed.author || '').toLowerCase().includes(authorName)) {
            if (filterMode === 'my' && !isFeedFollowed(feed)) return;
            return feed;
          }
      });
  } else if (searchQuery.startsWith("source:")) {
      var sourceDomain = searchQuery.replace("source:", "").trim().toLowerCase().replace(/^www\./, '');
      filteredFeeds = $.map(feeds, function(feed) {
          var feedDomain = extractDomainFromUrl(feed.link);
          if (feedDomain) {
              feedDomain = feedDomain.toLowerCase().replace(/^www\./, '');
              if (feedDomain.includes(sourceDomain)) {
                if (filterMode === 'my' && !isFeedFollowed(feed)) return;
                return feed;
              }
          }
      });
  } else {
      var searchWords = searchQuery.split(/\s+/);
      $.each(feeds, function(_, feed) {
          var title = (feed.title || '').toLowerCase();
          var description = (feed.description || '').toLowerCase();
          var has = searchWords.some(w => title.includes(w) || description.includes(w));
          if (has) {
            if (filterMode === 'my' && !isFeedFollowed(feed)) return;
            filteredFeeds.push(feed);
          }
      });
  }
  displayFeedsToContainer(filteredFeeds, $('#feeds'), true);
}
function displayFeedsAppend(shouldClear = false) {
  var searchQuery = $('#search-query').val().toLowerCase();
  var filteredFeeds = [];

  if (searchQuery.startsWith("author:")) {
      var authorName = searchQuery.replace("author:", "").trim();
      filteredFeeds = $.map(feeds, function(feed) {
          if ((feed.author || '').toLowerCase().includes(authorName)) {
            if (filterMode === 'my' && !isFeedFollowed(feed)) return;
            return feed;
          }
      });
  } else if (searchQuery.startsWith("source:")) {
      var sourceDomain = searchQuery.replace("source:", "").trim().toLowerCase().replace(/^www\./, '');
      filteredFeeds = $.map(feeds, function(feed) {
          var feedDomain = extractDomainFromUrl(feed.link);
          if (feedDomain) {
              feedDomain = feedDomain.toLowerCase().replace(/^www\./, '');
              if (feedDomain.includes(sourceDomain)) {
                if (filterMode === 'my' && !isFeedFollowed(feed)) return;
                return feed;
              }
          }
      });
  } else {
      var searchWords = searchQuery.split(/\s+/);
      $.each(feeds, function(_, feed) {
          var title = (feed.title || '').toLowerCase();
          var description = (feed.description || '').toLowerCase();
          var has = searchWords.some(w => title.includes(w) || description.includes(w));
          if (has) {
            if (filterMode === 'my' && !isFeedFollowed(feed)) return;
            filteredFeeds.push(feed);
          }
      });
  }

  displayFeedsToContainer(filteredFeeds, $('#feeds'), shouldClear);
}

/* ================== TAGS FROM SEARCH ================== */
function extractKeywordsAndDisplayTags(searchQuery) {
  var keywords = searchQuery.split(' ');
  var tagsContainer = document.getElementById('tags-container');
  tagsContainer.innerHTML = '';
  keywords.forEach(function(keyword) {
      var tag = document.createElement('a');
      tag.href = '#';
      if (keyword.length > 24) keyword = keyword.substring(0, 24) + '...';
      tag.textContent = '#' + keyword;

      var bgColor = stringToColor(keyword);
      var textColor = isColorDark(bgColor) ? 'white' : 'black';
      tag.style.backgroundColor = bgColor;
      tag.style.color = textColor;
      tag.style.border = '2px solid black';
      tag.style.padding = '4px 8px';
      tag.style.marginRight = '5px';
      tag.style.borderRadius = '5px';
      tag.style.textDecoration = 'none';
      tag.style.display = 'inline-block';
      tag.style.marginBottom = '5px';
      tag.style.fontWeight = 'bold';

      tag.classList.add('tag-search-link');
      tag.dataset.tag = keyword;
      tagsContainer.appendChild(tag);
  });
}
function handleSearch() {
  var searchQuery = $('#search-query').val().toLowerCase();
  extractKeywordsAndDisplayTags(searchQuery);
  page = 1;
  if (window.masonry) window.masonry.clear(); else $('#feeds').empty();
  loadFeeds();
}
$('#search-button').click(handleSearch);
$('#search-query').keypress(function(e){ if (e.which === 13){ e.preventDefault(); handleSearch(); } });

/* ================== RENDER (cards) ================== */
function displayFeedsToContainer(feedsArr, container, clearFirst = true) {
  if (clearFirst) {
      if (window.masonry) { window.masonry.clear(); }
      else { container.empty(); }
      displayedFeeds = {};
  }

  feedsArr.sort(function(a, b) {
      var dateA = new Date(a.pubDate);
      var dateB = new Date(b.pubDate);
      return dateA < dateB ? 1 : -1;
  });

  $.each(feedsArr, function(index, feed) {
      if (displayedFeeds[feed.link]) return;
      displayedFeeds[feed.link] = true;
      feedsById[feed.id] = feed;

      var thumbnail = feed.thumbnail || '';
      var author = feed.author || 'Anonymous';

      var sourceName = feed.sourceName || extractDomainFromUrl(feed.link) || 'Unknown';
      var sourceLogo = feed.sourceLogo;
      var sourceDomain = extractBaseDomainFromUrl(feed.link) || feed.source_domain || '';
      var truncatedSourceName = truncateText(sourceName, 24);
      var channelAccent = accColorFromText(sourceDomain);

      var title = feed.title || 'Untitled';
      var description = feed.description || '';
      var pubDate = feed.pubDate || '';
      var truncatedTitle = truncateText(title, 80);
      var truncatedDescription = truncateText(description, 80);

      // media
      var mediaHtml = '';
      if (sourceDomain.includes('facebook.com'))      mediaHtml = generateFacebookEmbed(feed.link);
      else if (sourceDomain.includes('tiktok.com'))   mediaHtml = generateTikTokEmbed(feed.link);
      else if (sourceDomain.includes('instagram.com'))mediaHtml = generateInstagramEmbed(feed.link);
      else if (sourceDomain.includes('youtube.com') || sourceDomain.includes('youtu.be')) mediaHtml = generateYouTubeThumbnailEmbed(feed.link);
      else if (sourceDomain.includes('t.me'))         mediaHtml = generateTelegramEmbed(feed.link);
      else mediaHtml = thumbnail ? `<a href="pindetails.php?id=${feed.id}" target="_blank" rel="noopener noreferrer"><img src="${thumbnail}" alt="Thumbnail" class="feed-thumbnail"></a>` : '';

      // site-strip
      var faviconLink = feed.link ? new URL(feed.link).origin + '/favicon.ico' : '';
      var siteStrip = `
        <div class="site-strip">
           <img src="${faviconLink}" alt="favicon">
           <a href="${feed.link}" target="_blank" rel="noopener" class="site-name">${truncatedSourceName}</a>
        </div>`;

      // tags
      var tags = extractTagsFromTitle(title);
      var tagsHtml = '<div class="tags-row">';
      if (tags.length > 0) {
          tags.forEach((tag) => {
              var rawTag = tag;
              var short = truncateText(tag, 28);
              let chipBg = stringToColor(rawTag);
              let chipText = isColorDark(chipBg) ? 'white' : 'black';
              var isTagFollowed = userFollowSets.tags.has((rawTag||'').toLowerCase());
              var iconClass = isTagFollowed ? 'fa-check' : 'fa-plus';
              var followingClass = isTagFollowed ? 'following' : '';
              var titleTip = isTagFollowed ? 'Unfollow tag' : 'Follow tag';
              tagsHtml += `
                <span class="tag-chip">
                  <a href="#" class="tag-search-link" data-tag="${rawTag}" style="background-color:${chipBg};color:${chipText};">#${short}</a>
                  <button class="follow-icon ${followingClass} tag-follow" data-tag="${encodeURIComponent(rawTag)}" aria-label="${titleTip}" title="${titleTip}">
                    <i class="fas ${iconClass}"></i>
                  </button>
                </span>`;
          });
      }
      tagsHtml += '</div>';

      // header author
      var authorAvatar = thumbnail
          ? `<img src="${thumbnail}" alt="Avatar" class="author-avatar" style="width:30px;height:30px;margin-right:5px;border:2px solid black;border-radius:30px;vertical-align:middle;object-fit:cover;">`
          : '';
      var isFollowingAuthor = userFollowSets.authors.has((author||'').toLowerCase());
      var followBtnClass = isFollowingAuthor ? 'follow-btn following' : 'follow-btn';
      var followIcon = isFollowingAuthor ? 'fa-check' : 'fa-plus';
      var followLabel = isFollowingAuthor ? 'Following' : 'Follow';
      var authorProfileUrl = `author.php?name=${encodeURIComponent(author)}`; /* UPDATED */
      var authorTitle = isFollowingAuthor ? 'Unfollow author' : 'Follow author';

      var cardHeader = `
        <div class="card-header">
          <div class="author-block">
            <a href="${authorProfileUrl}" class="author-avatar-link" title="@${author}">${authorAvatar}</a>
            <a href="#" class="author-search-link author-name" data-author="${author}">@${author}</a>
          </div>
          <button class="${followBtnClass}" data-type="author" data-value="${encodeURIComponent(author)}" title="${authorTitle}">
            <i class="fas ${followIcon}"></i> <span class="label">${followLabel}</span>
          </button>
        </div>`;

      // channel + follow (icon-only)
      var isFollowingDomain = userFollowSets.sourceDomains.has((sourceDomain||'').toLowerCase());
      var domIconClass = isFollowingDomain ? 'fa-check' : 'fa-plus';
      var domFollowCls = isFollowingDomain ? 'following' : '';
      var domTitle = isFollowingDomain ? 'Unfollow channel' : 'Follow channel';
      var channelRow = `
        <div class="channel-row" style="--channel-accent:${channelAccent}">
          <a href="#" class="channel-left source-search-link" data-source-domain="${sourceDomain}" title="Search channel">
            <span class="badge"><i class="fas fa-globe"></i> Channel</span>
            ${sourceLogo ? `<img src="${sourceLogo}" alt="${sourceName}" style="width:18px;height:18px;border-radius:4px;">` : ``}
            <span class="source-name">${truncatedSourceName}</span>
          </a>
          <button class="follow-icon ${domFollowCls} follow-channel" data-type="source_domain" data-value="${encodeURIComponent(sourceDomain)}" title="${domTitle}">
            <i class="fas ${domIconClass}"></i>
          </button>
        </div>`;

      // actions
      var actionsRow = `
        <div class="actions-row">
          <div class="actions-left">
            <button class="action-btn like-btn" data-feed-id="${feed.id}" title="Like">
              <i class="fas fa-thumbs-up"></i><span class="label">Like</span> <span class="count like-count">0</span>
            </button>
            <button class="action-btn dislike-btn" data-feed-id="${feed.id}" title="Dislike">
              <i class="fas fa-thumbs-down"></i><span class="label">Dislike</span> <span class="count dislike-count">0</span>
            </button>
          </div>
          <div class="actions-right">
            <button class="action-btn copy-embed-btn" data-feed-id="${feed.id}" title="Copy embed code">
              <i class="fas fa-code"></i><span class="label">Embed</span>
            </button>
            <button class="action-btn save-btn" data-feed-id="${feed.id}" title="Save to favorites">
              <i class="fas fa-bookmark"></i><span class="label">Save</span>
            </button>
          </div>
        </div>`;

      var html = `
        <div class="result">
          ${cardHeader}
          ${mediaHtml}
          <a href="pindetails.php?id=${feed.id}" target="_blank" rel="noopener noreferrer"><h2>${truncatedTitle}</h2></a>
          <p>${truncatedDescription}</p>
          ${channelRow}
          ${tagsHtml}
          ${siteStrip}
          <p>${pubDate}</p>
          ${actionsRow}
          <div class="share-buttons-container">${createShareButton(feed)}</div>
        </div>`;

      var $item = $(html);
      container.append($item);

      // Restore state (likes/dislikes/saved)
      var feedId = feed.id;
      $.getJSON('api_likes.php', { feed_id: feedId }, function(res){
        if (res && res.success){
          $item.find('.like-count').text(res.likes);
          $item.find('.dislike-count').text(res.dislikes);
          var liked = res.user_reaction === 'like';
          var disliked = res.user_reaction === 'dislike';
          $item.find('.like-btn').toggleClass('liked', liked)
               .attr('title', liked ? 'Remove like' : 'Like')
               .find('.label').text(liked ? 'Liked' : 'Like');
          $item.find('.dislike-btn').toggleClass('disliked', disliked)
               .attr('title', disliked ? 'Remove dislike' : 'Dislike')
               .find('.label').text(disliked ? 'Disliked' : 'Dislike');
        }
      });
      $.getJSON('api_favorites.php', { feed_id: feedId }, function(res){
        if (res && res.success && res.favorited){
          $item.find('.save-btn').addClass('saved')
               .attr('title','Remove from favorites')
               .find('.label').text('Saved');
        }
      });

      if (window.masonry) { setTimeout(() => { window.masonry.addItem($item[0]); }, index * 15); }
  });

  if (typeof FB !== 'undefined') { FB.XFBML.parse(); }
  else { setTimeout(function(){ if (typeof FB !== 'undefined') FB.XFBML.parse(); }, 1000); }

  /* ================== EVENTS ================== */

  // like
  $(document).off('click.like').on('click.like', '.like-btn', function(){
    var $btn = $(this), id = $btn.data('feed-id');
    $.ajax({
      url: 'api_likes.php', method: 'POST', contentType:'application/json',
      data: JSON.stringify({ feed_id:id, action:'like' }),
      success: function(res){
        if (res.success){
          $btn.find('.like-count').text(res.likes);
          $(`.dislike-btn[data-feed-id="${id}"] .dislike-count`).text(res.dislikes);

          var liked = res.user_reaction==='like';
          var disliked = res.user_reaction==='dislike';

          $btn.toggleClass('liked', liked)
              .attr('title', liked ? 'Remove like' : 'Like')
              .find('.label').text(liked ? 'Liked' : 'Like');

          var $d = $(`.dislike-btn[data-feed-id="${id}"]`);
          $d.toggleClass('disliked', disliked)
            .attr('title', disliked ? 'Remove dislike' : 'Dislike')
            .find('.label').text(disliked ? 'Disliked' : 'Dislike');

          showLikeToast(res.action!=='removed', 'like');
        } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
      }
    });
  });

  // dislike
  $(document).off('click.dislike').on('click.dislike', '.dislike-btn', function(){
    var $btn = $(this), id = $btn.data('feed-id');
    $.ajax({
      url: 'api_likes.php', method: 'POST', contentType:'application/json',
      data: JSON.stringify({ feed_id:id, action:'dislike' }),
      success: function(res){
        if (res.success){
          $(`.like-btn[data-feed-id="${id}"] .like-count`).text(res.likes);
          $btn.find('.dislike-count').text(res.dislikes);

          var liked = res.user_reaction==='like';
          var disliked = res.user_reaction==='dislike';

          var $l = $(`.like-btn[data-feed-id="${id}"]`);
          $l.toggleClass('liked', liked)
            .attr('title', liked ? 'Remove like' : 'Like')
            .find('.label').text(liked ? 'Liked' : 'Like');

          $btn.toggleClass('disliked', disliked)
              .attr('title', disliked ? 'Remove dislike' : 'Dislike')
              .find('.label').text(disliked ? 'Disliked' : 'Dislike');

          showLikeToast(res.action!=='removed', 'dislike');
        } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
      }
    });
  });

  // save (favorites)
  $(document).off('click.save').on('click.save', '.save-btn', function(){
    var $btn = $(this), id = $btn.data('feed-id');
    $.ajax({
      url: 'api_favorites.php', method: 'POST', contentType:'application/json',
      data: JSON.stringify({ feed_id:id }),
      success: function(res){
        if (res.success){
          var saved = !!res.favorited;
          $btn.toggleClass('saved', saved)
              .attr('title', saved ? 'Remove from favorites' : 'Save to favorites')
              .find('.label').text(saved ? 'Saved' : 'Save');
          showSaveToast(saved);
          if (saved) fireConfetti('#40c4ff');
        } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
      }
    });
  });

  // COPY IFRAME (Embed)
  $(document).off('click.copy').on('click.copy', '.copy-embed-btn', function(){
    var id = $(this).data('feed-id');
    var feed = feedsById[id] || null;
    if (!feed) return;
    var code = generateIframeCode(feed);

    if (navigator.clipboard && navigator.clipboard.writeText) {
      navigator.clipboard.writeText(code).then(function(){
        showEmbedToast();
      }).catch(function(){
        var ta = document.createElement('textarea');
        ta.value = code; document.body.appendChild(ta); ta.select();
        try { document.execCommand('copy'); showEmbedToast(); } catch(e){}
        document.body.removeChild(ta);
      });
    } else {
      var ta = document.createElement('textarea');
      ta.value = code; document.body.appendChild(ta); ta.select();
      try { document.execCommand('copy'); showEmbedToast(); } catch(e){}
      document.body.removeChild(ta);
    }
  });

  // follow author (pill) + confirm & hover “Unfollow”
  $(document).off('mouseenter.followhover').on('mouseenter.followhover', '.follow-btn.following', function(){
    var $b = $(this);
    if (!$b.data('orig-label')) $b.data('orig-label', $b.find('.label').text());
    $b.find('.label').text('Unfollow');
    var val = decodeURIComponent($b.data('value') || '');
    $b.attr('title', 'Unfollow @' + val);
  });
  $(document).off('mouseleave.followhover').on('mouseleave.followhover', '.follow-btn.following', function(){
    var $b = $(this);
    var orig = $b.data('orig-label') || 'Following';
    $b.find('.label').text(orig);
    $b.attr('title', 'Unfollow author');
  });
  $(document).off('click.follow-author').on('click.follow-author', '.follow-btn', function(){
    var $btn = $(this), value = decodeURIComponent($btn.data('value'));
    var isFollowing = $btn.hasClass('following');
    function perform(){
      $.ajax({
        url:'api_follow.php', method:'POST', contentType:'application/json',
        data: JSON.stringify({ follow_type:'author', follow_value:value }),
        success: function(res){
          if (res.success){
            var followed = !!res.following;
            $btn.toggleClass('following', followed);
            $btn.find('i').toggleClass('fa-check', followed).toggleClass('fa-plus', !followed);
            $btn.find('.label').text(followed ? 'Following' : 'Follow');
            $btn.attr('title', followed ? 'Unfollow author' : 'Follow author');
            if (followed) userFollowSets.authors.add(value.toLowerCase()); else userFollowSets.authors.delete(value.toLowerCase());
            showFollowToast('author', value, followed);
            if (followed) fireConfetti(accColorFromText(value));
          } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
        }
      });
    }
    if (isFollowing) confirmUnfollow('author', value, perform); else perform();
  });

  // follow channel (icon only)
  $(document).off('click.follow-channel').on('click.follow-channel', '.follow-channel', function(){
    var $btn = $(this), value = decodeURIComponent($btn.data('value'));
    var isFollowing = $btn.hasClass('following');
    function perform(){
      $.ajax({
        url:'api_follow.php', method:'POST', contentType:'application/json',
        data: JSON.stringify({ follow_type:'source_domain', follow_value:value }),
        success: function(res){
          if (res.success){
            var followed = !!res.following;
            $btn.toggleClass('following', followed);
            $btn.find('i').toggleClass('fa-check', followed).toggleClass('fa-plus', !followed);
            $btn.attr('title', followed ? 'Unfollow channel' : 'Follow channel');
            if (followed) userFollowSets.sourceDomains.add(value.toLowerCase()); else userFollowSets.sourceDomains.delete(value.toLowerCase());
            showFollowToast('source_domain', value, followed);
            if (followed) fireConfetti(accColorFromText(value));
          } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
        }
      });
    }
    if (isFollowing) confirmUnfollow('source_domain', value, perform); else perform();
  });

  // follow tag (icon only)
  $(document).off('click.follow-tag').on('click.follow-tag', '.tag-follow', function(){
    var $btn = $(this), tag = decodeURIComponent($btn.data('tag'));
    var isFollowing = $btn.hasClass('following');
    function perform(){
      $.ajax({
        url:'api_follow.php', method:'POST', contentType:'application/json',
        data: JSON.stringify({ follow_type:'tag', follow_value:tag }),
        success: function(res){
          if (res.success){
            var followed = !!res.following;
            $btn.toggleClass('following', followed);
            $btn.find('i').toggleClass('fa-check', followed).toggleClass('fa-plus', !followed);
            $btn.attr('title', followed ? 'Unfollow tag' : 'Follow tag');
            if (followed) userFollowSets.tags.add(tag.toLowerCase()); else userFollowSets.tags.delete(tag.toLowerCase());
            showFollowToast('tag', tag, followed);
            if (followed) fireConfetti(accColorFromText(tag));
          } else if (res.error==='not_logged_in'){ window.location.href='login.html'; }
        }
      });
    }
    if (isFollowing) confirmUnfollow('tag', tag, perform); else perform();
  });

  // AUTHOR SEARCH CLICK -> open author page (UPDATED)
  $(document).off('click.author-search').on('click.author-search', '.author-search-link', function(e){
    e.preventDefault();
    var author = $(this).data('author');
    window.location.href = 'author.php?name=' + encodeURIComponent(author);
  });

  // Domain (channel) search
  $(document).off('click.source-search').on('click.source-search', '.source-search-link', function(e){
    e.preventDefault();
    var domain = $(this).data('source-domain');
    $('#search-query').val('source:' + domain);
    page = 1; window.scrollTo({ top: 0, behavior: 'smooth' }); loadFeeds();
  });

  // YouTube thumb -> iframe
  $(document).off('click.ytthumb').on('click.ytthumb', '.youtube-thumbnail', function(){
    const embedURL = $(this).data('embed-url');
    const html = `<iframe src="${embedURL}" style="width: 100%; height: 100%; border: none;" frameborder="0" allow="accelerometer; autoplay; encrypted-media; gyroscope; picture-in-picture" allowfullscreen></iframe>`;
    $(this).replaceWith(html);
  });
}

/* ================== EMBEDS ================== */
function generateTelegramEmbed(url) {
  const embedURL = `https://t.me/${extractTelegramChannelAndMessageId(url)}?embed=1`;
  return `
    <div style="position: relative; padding-bottom: 56.25%; height: 0; overflow: hidden; max-width: 100%;">
      <iframe src="${embedURL}" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: none; overflow: hidden;" scrolling="yes" frameborder="0" allowTransparency="true" allow="encrypted-media"></iframe>
    </div>`;
}
function extractTelegramChannelAndMessageId(url) { const m = url.match(/t\.me\/([^/]+)\/(\d+)/); return m ? `${m[1]}/${m[2]}` : null; }
function generateFacebookEmbed(url) {
  const embedURL = `https://www.facebook.com/plugins/post.php?href=${encodeURIComponent(url)}&width=500`;
  return `
    <div style="position: relative; padding-bottom: 56.25%; height: 0; overflow: hidden; max-width: 100%;">
      <iframe src="${embedURL}" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: none; overflow: hidden;" scrolling="yes" frameborder="0" allowTransparency="true" allow="encrypted-media"></iframe>
    </div>`;
}
function generateTikTokEmbed(url) {
  const embedURL = `https://www.tiktok.com/embed/v2/${extractTikTokVideoId(url)}`;
  return `
    <div style="position: relative; padding-bottom: 177.78%; height: 0; overflow: hidden; max-width: 100%;">
      <iframe src="${embedURL}" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: none; overflow: hidden;" scrolling="yes" frameborder="0" allowTransparency="true" allow="encrypted-media"></iframe>
    </div>`;
}
function generateInstagramEmbed(url) {
  const embedURL = `https://www.instagram.com/p/${extractInstagramPostId(url)}/embed`;
  return `
    <div style="position: relative; padding-bottom: 56.25%; height: 0; overflow: hidden; max-width: 100%;">
      <iframe src="${embedURL}" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: none; overflow: hidden;" scrolling="yes" frameborder="0" allowTransparency="true" allow="encrypted-media"></iframe>
    </div>`;
}
function generateYouTubeThumbnailEmbed(url) {
  const videoId = extractYouTubeVideoId(url);
  const thumbnailURL = `https://img.youtube.com/vi/${videoId}/hqdefault.jpg`;
  const embedURL = `https://www.youtube.com/embed/${videoId}`;
  return `
    <div class="youtube-thumbnail" data-embed-url="${embedURL}" style="position: relative; cursor: pointer;">
      <img src="${thumbnailURL}" alt="YouTube Thumbnail" style="width: 100%; height: auto;">
      <div class="play-button" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 64px; color: white; text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.8);">▶</div>
    </div>`;
}
function extractTikTokVideoId(url) { const m = url.match(/tiktok\.com\/(?:@[^/]+\/video\/|v\/)(\d+)/); return m ? m[1] : null; }
function extractInstagramPostId(url) { const m = url.match(/instagram\.com\/p\/([^/]+)/); return m ? m[1] : null; }
function extractYouTubeVideoId(url) { const m = url.match(/(?:youtube\.com\/(?:[^/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^" &?/]+)/); return m ? m[1] : null; }

/* ================== COPY IFRAME (generator) ================== */
function generateIframeCode(feed) {
  var author = feed.author || 'Anonymous';
  var authorAvatar = feed.thumbnail || 'https://myhashtag.io/static/default-avatar.png';
  var sourceName = feed.sourceName || extractDomainFromUrl(feed.link) || 'Source';
  var faviconLink = feed.link ? new URL(feed.link).origin + '/favicon.ico' : '';
  var tags = extractTagsFromTitle(feed.title || '');

  var tagsHtml = '';
  if (tags.length > 0) {
    tagsHtml = '<div class="mh-tags">';
    tags.forEach(function(t){
      var short = t.length > 28 ? (t.substring(0, 28) + '…') : t;
      var bg = stringToColor(t);
      var color = isColorDark(bg) ? '#fff' : '#000';
      tagsHtml += `<span class="mh-tag" style="background:${bg};color:${color};border:2px solid #000;">#${short}</span>`;
    });
    tagsHtml += '</div>';
  }

  var inner = `
  <div class="mh-embed">
    <a class="mh-thumb" href="https://myhashtag.io/pindetails.php?id=${feed.id}" target="_blank" rel="noopener">
      ${feed.thumbnail ? `<img src="${feed.thumbnail}" alt="thumb">` : ''}
    </a>
    <div class="mh-body">
      <a class="mh-title" href="https://myhashtag.io/pindetails.php?id=${feed.id}" target="_blank" rel="noopener">${(feed.title||'').replace(/</g,'&lt;').replace(/>/g,'&gt;')}</a>
      <div class="mh-desc">${(feed.description||'').replace(/</g,'&lt;').replace(/>/g,'&gt;')}</div>

      <div class="mh-author">
        <img class="mh-avatar" src="${authorAvatar}" alt="avatar">
        <span class="mh-aname">@${author}</span>
      </div>

      ${tagsHtml}

      <a class="mh-site" href="${feed.link}" target="_blank" rel="noopener">
        <img class="mh-fav" src="${faviconLink}" alt="fav">
        <span class="mh-sname">${sourceName}</span>
      </a>
    </div>
  </div>`;

  var css = `
  <style>
    *{box-sizing:border-box}
    body{margin:0;font-family:Arial,Helvetica,sans-serif;background:#0b1220;color:#e5edf7}
    .mh-embed{width:100%;max-width:320px;background:linear-gradient(180deg,#33475b,#2f4254);border:1px solid rgba(255,255,255,.08);border-radius:12px;overflow:hidden}
    .mh-thumb img{display:block;width:100%;height:auto}
    .mh-body{padding:8px 10px 10px}
    .mh-title{display:block;font-weight:800;font-size:14px;color:#fff;text-decoration:none;margin:6px 0 6px 0}
    .mh-title:hover{filter:brightness(1.05)}
    .mh-desc{font-size:12px;color:#cdd7e4;line-height:1.3;margin-bottom:6px;display:-webkit-box;-webkit-line-clamp:2;-webkit-box-orient:vertical;overflow:hidden}
    .mh-author{display:flex;align-items:center;gap:6px;margin:6px 0}
    .mh-avatar{width:24px;height:24px;border-radius:50%;object-fit:cover;border:2px solid #000}
    .mh-aname{font-weight:700;font-size:12.5px}
    .mh-tags{display:flex;flex-wrap:wrap;gap:6px;margin:6px 0}
    .mh-tag{display:inline-block;padding:2px 6px;border-radius:6px;font-weight:800;font-size:11.5px}
    .mh-site{display:flex;align-items:center;gap:8px;background:rgba(255,255,255,.05);border-radius:8px;padding:6px 8px;text-decoration:none;color:#e5edf7;border:1px solid rgba(255,255,255,.1)}
    .mh-site:hover{filter:brightness(1.03)}
    .mh-fav{width:16px;height:16px;border-radius:3px}
    .mh-sname{font-size:12.5px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis}
  </style>`;

  var srcdoc = (css + inner).replace(/"/g,'&quot;').replace(/'/g,'&#39;');
  return `<iframe srcdoc="${srcdoc}" width="320" height="560" frameborder="0" style="border:0;border-radius:12px;overflow:hidden;"></iframe>`;
}

/* ================== INFINITE SCROLL ================== */
$(window).scroll(function() {
  if ($(window).scrollTop() + $(window).height() > $(document).height() - loadThreshold) {
      if (!loading) {
        if (window.__AUTHOR_PAGE__ && typeof window.loadAuthorFeeds === 'function') {
          page++; window.loadAuthorFeeds();
        } else {
          page++; loadFeeds();
        }
      }
  }
});

/* ================== LOADER (main) ================== */
function loadFeeds() {
  if (loading) return;
  loading = true;

  ensureFilterBar();

  const loadingIndicator = document.getElementById('loadingIndicator');
  if (loadingIndicator) loadingIndicator.style.display = 'flex';

  var searchTerm = $('#search-query').val();
  var timestamp = new Date().getTime();

  // Choose endpoint based on filter
  var endpoint = (filterMode === 'my') ? 'get_pinfeeds_following.php' : 'get_pinfeeds.php';

  $.getJSON(endpoint, { page: page, search: searchTerm, t: timestamp }, function(data) {
      if (page === 1) {
          feeds = {};
          feedsById = {};
          displayedFeeds = {};
          if (window.masonry) window.masonry.clear(); else $('#feeds').empty();
      }

      $.each(data, function(index, feed) {
        feeds[feed.link] = feed;
        feedsById[feed.id] = feed;
      });

      loading = false;
      if (loadingIndicator) loadingIndicator.style.display = 'none';

      if (!window.masonry) {
          const container = document.querySelector('#feeds');
          if (container) window.masonry = new MasonryLayout(container, { gap: 20, columns: 10 });
      }

      var shouldClear = (page === 1);

      fetchUserFollowSetsOnce().then(function(){
          if (endpoint === 'get_pinfeeds.php') {
              if (searchTerm === '') displayFeedsOnlyWithImageAppend(shouldClear);
              else displayFeedsAppend(shouldClear);
          } else {
              var arr = Array.isArray(data) ? data : [];
              if (!searchTerm) arr = arr.filter(f => (f && (f.image || f.thumbnail)));
              displayFeedsToContainer(arr, $('#feeds'), shouldClear);
          }
      });

  }).fail(function(jqXHR, textStatus, errorThrown) {
      console.error("Error loading feeds:", textStatus, errorThrown);
      alert('Unable to load feeds. Please try again.');
      loading = false;
      if (loadingIndicator) loadingIndicator.style.display = 'none';
  });
}

/* Inicial (guard on author page) */
if (!window.__AUTHOR_PAGE__) { loadFeeds(); }