/**
 * Sistema de Correção Completo para Masonry Layout
 * Este arquivo deve ser carregado APÓS o feeds.js original
 * 
 * Funcionalidades:
 * - Reorganização suave durante resize
 * - Correção automática de problemas de layout
 * - Suporte completo para mobile
 * - Debug avançado
 * - Performance otimizada
 */

// CORREÇÃO COMPLETA DO MASONRY - Adicionar DEPOIS do feeds.js original

(function() {
    'use strict';

    console.log('[MasonryFix] Iniciando sistema de correção...');

    // Adiciona CSS crítico para o funcionamento
    const criticalCSS = `
        /* CSS Crítico para Masonry - Correções */
        .result-list {
            position: relative !important;
            transition: height 0.5s cubic-bezier(0.4, 0, 0.2, 1);
            min-height: 100px;
            opacity: 1 !important;
            visibility: visible !important;
        }
        
        .result {
            transition: none;
            backface-visibility: hidden;
            -webkit-backface-visibility: hidden;
            will-change: transform, left, top, width;
            transform: translateZ(0);
            -webkit-transform: translateZ(0);
        }
        
        /* Classe para quando estiver reorganizando */
        .result.transitioning {
            transition: left 0.5s cubic-bezier(0.4, 0, 0.2, 1), 
                        top 0.5s cubic-bezier(0.4, 0, 0.2, 1), 
                        width 0.5s cubic-bezier(0.4, 0, 0.2, 1) !important;
        }
        
        .result-list.resizing {
            opacity: 0.95;
        }
        
        /* Mobile - força layout em coluna única */
        @media (max-width: 500px) {
            .result {
                position: relative !important;
                width: 100% !important;
                left: auto !important;
                top: auto !important;
                margin-bottom: 15px;
                transition: none !important;
                display: block !important;
                opacity: 1 !important;
                visibility: visible !important;
            }
            
            .result-list {
                height: auto !important;
                display: block !important;
            }
        }
        
        /* Garantir visibilidade durante carregamento */
        .result.loading {
            opacity: 0.5;
        }
        
        .result.error {
            border: 2px solid red;
        }
    `;
    
    // Injeta CSS
    const styleElement = document.createElement('style');
    styleElement.id = 'masonry-fix-styles';
    styleElement.textContent = criticalCSS;
    document.head.appendChild(styleElement);

    // Aguarda a classe MasonryLayout estar disponível
    const waitForMasonry = setInterval(() => {
        if (window.MasonryLayout) {
            clearInterval(waitForMasonry);
            applyMasonryFixes();
        }
    }, 100);

    function applyMasonryFixes() {
        console.log('[MasonryFix] Aplicando correções na classe MasonryLayout...');

        // Salvamos referência aos métodos originais
        const OriginalMasonryLayout = window.MasonryLayout;
        const originalRelayout = OriginalMasonryLayout.prototype.relayout;
        const originalAddItem = OriginalMasonryLayout.prototype.addItem;
        const originalBindEvents = OriginalMasonryLayout.prototype.bindEvents;

        // Método relayout melhorado
        OriginalMasonryLayout.prototype.relayout = async function() {
            console.log('[MasonryFix] Executando relayout melhorado...');
            
            const startTime = performance.now();
            const elements = Array.from(this.container.querySelectorAll('.result'));
            
            if (elements.length === 0) {
                console.log('[MasonryFix] Nenhum elemento para reorganizar');
                return;
            }
            
            // Verifica se é mobile
            const isMobile = this.columns === 1 || window.innerWidth <= 500;
            
            if (isMobile) {
                console.log('[MasonryFix] Modo mobile - aplicando layout simples');
                this.container.classList.add('mobile-layout');
                elements.forEach(element => {
                    element.classList.remove('transitioning');
                    element.style.cssText = `
                        position: relative !important;
                        width: 100% !important;
                        left: auto !important;
                        top: auto !important;
                        margin-bottom: 15px;
                        opacity: 1 !important;
                        visibility: visible !important;
                    `;
                });
                this.container.style.height = 'auto';
                return;
            }
            
            // Remove classe mobile se estava presente
            this.container.classList.remove('mobile-layout');
            
            // Indica que está reorganizando
            this.container.classList.add('resizing');
            
            console.log(`[MasonryFix] Reorganizando ${elements.length} elementos em ${this.columns} colunas`);
            
            // Reset das alturas das colunas
            this.columnHeights = new Array(this.columns).fill(0);
            
            // Calcula dimensões
            const containerWidth = this.container.offsetWidth;
            const gap = this.gap || 20;
            const itemWidth = Math.floor((containerWidth - (this.columns - 1) * gap) / this.columns);
            
            console.log(`[MasonryFix] Dimensões: Container=${containerWidth}px, Item=${itemWidth}px, Gap=${gap}px`);
            
            // Adiciona classe de transição
            elements.forEach(el => {
                el.classList.add('transitioning');
                el.style.visibility = 'visible';
                el.style.opacity = '1';
            });
            
            // Aguarda um frame
            await new Promise(resolve => requestAnimationFrame(resolve));
            
            // Array para armazenar promessas de imagens
            const imagePromises = [];
            
            // Processa cada elemento
            for (let i = 0; i < elements.length; i++) {
                const element = elements[i];
                
                // Carrega imagens
                const images = element.querySelectorAll('img');
                images.forEach(img => {
                    if (!img.complete) {
                        imagePromises.push(new Promise(resolve => {
                            img.onload = resolve;
                            img.onerror = resolve;
                            // Timeout de 2 segundos
                            setTimeout(resolve, 2000);
                        }));
                    }
                });
            }
            
            // Aguarda todas as imagens
            if (imagePromises.length > 0) {
                console.log(`[MasonryFix] Aguardando ${imagePromises.length} imagens...`);
                await Promise.all(imagePromises);
            }
            
            // Reorganiza elementos
            for (let i = 0; i < elements.length; i++) {
                const element = elements[i];
                
                // Força recálculo de altura
                element.style.height = 'auto';
                const elementHeight = element.offsetHeight;
                
                // Encontra coluna mais curta
                let shortestColumn = 0;
                let minHeight = this.columnHeights[0];
                
                for (let j = 1; j < this.columns; j++) {
                    if (this.columnHeights[j] < minHeight) {
                        minHeight = this.columnHeights[j];
                        shortestColumn = j;
                    }
                }
                
                // Calcula posição
                const left = shortestColumn * (itemWidth + gap);
                const top = this.columnHeights[shortestColumn];
                
                // Aplica estilos
                element.style.cssText = `
                    position: absolute;
                    width: ${itemWidth}px;
                    left: ${left}px;
                    top: ${top}px;
                    opacity: 1;
                    visibility: visible;
                `;
                
                // Preserva classes e transições
                element.classList.add('transitioning');
                element.classList.add('show');
                
                // Atualiza altura da coluna
                this.columnHeights[shortestColumn] += elementHeight + gap;
                
                if (i % 10 === 0) {
                    console.log(`[MasonryFix] Processados ${i + 1}/${elements.length} elementos`);
                }
            }
            
            // Atualiza altura do container
            const maxHeight = Math.max(...this.columnHeights);
            this.container.style.height = maxHeight + 'px';
            
            const endTime = performance.now();
            console.log(`[MasonryFix] Relayout concluído em ${(endTime - startTime).toFixed(2)}ms`);
            console.log(`[MasonryFix] Altura final: ${maxHeight}px`);
            console.log(`[MasonryFix] Alturas das colunas:`, this.columnHeights);
            
            // Remove classes de transição após animação
            setTimeout(() => {
                elements.forEach(el => el.classList.remove('transitioning'));
                this.container.classList.remove('resizing');
            }, 600);
        };

        // Método addItem melhorado
        OriginalMasonryLayout.prototype.addItem = async function(element) {
            // Se for mobile, usa layout simples
            if (this.columns === 1 || window.innerWidth <= 500) {
                element.style.cssText = `
                    position: relative !important;
                    width: 100% !important;
                    left: auto !important;
                    top: auto !important;
                    margin-bottom: 15px;
                    opacity: 1 !important;
                    visibility: visible !important;
                `;
                element.classList.add('show');
                return;
            }
            
            // Marca como carregando
            element.classList.add('loading');
            
            try {
                // Aguarda imagens
                await this.waitForImages(element);
                
                // Encontra coluna mais curta
                let shortestColumn = 0;
                let minHeight = this.columnHeights[0];
                
                for (let i = 1; i < this.columns; i++) {
                    if (this.columnHeights[i] < minHeight) {
                        minHeight = this.columnHeights[i];
                        shortestColumn = i;
                    }
                }
                
                // Calcula dimensões
                const containerWidth = this.container.offsetWidth;
                const itemWidth = (containerWidth - (this.columns - 1) * this.gap) / this.columns;
                
                // Define posição inicial (com offset para animação)
                element.style.cssText = `
                    position: absolute;
                    width: ${itemWidth}px;
                    left: ${shortestColumn * (itemWidth + this.gap)}px;
                    top: ${this.columnHeights[shortestColumn] + 20}px;
                    opacity: 0;
                    transform: translateY(20px);
                    visibility: visible;
                `;
                
                // Força reflow
                element.offsetHeight;
                
                // Adiciona transição
                element.style.transition = 'opacity 0.4s ease, transform 0.4s ease, top 0.4s ease';
                
                // Anima entrada
                requestAnimationFrame(() => {
                    element.style.opacity = '1';
                    element.style.transform = 'translateY(0)';
                    element.style.top = this.columnHeights[shortestColumn] + 'px';
                    element.classList.add('show');
                    element.classList.remove('loading');
                });
                
                // Atualiza altura da coluna
                const elementHeight = element.offsetHeight;
                this.columnHeights[shortestColumn] += elementHeight + this.gap;
                
                // Atualiza altura do container
                const maxHeight = Math.max(...this.columnHeights);
                this.container.style.height = maxHeight + 'px';
                
                // Marca container como carregado
                this.container.classList.add('loaded');
                
                // Remove transição após animação
                setTimeout(() => {
                    element.style.transition = '';
                }, 400);
                
            } catch (error) {
                console.error('[MasonryFix] Erro ao adicionar item:', error);
                element.classList.add('error');
                element.classList.remove('loading');
            }
        };

        // Método bindEvents melhorado
        OriginalMasonryLayout.prototype.bindEvents = function() {
            let previousWidth = window.innerWidth;
            let previousColumns = this.columns;
            let resizeTimer = null;
            let isResizing = false;
            
            const handleResize = () => {
                const currentWidth = window.innerWidth;
                const widthDifference = Math.abs(currentWidth - previousWidth);
                
                // Atualiza colunas
                this.updateColumns();
                const columnsChanged = this.columns !== previousColumns;
                
                // Só reorganiza se necessário
                if (widthDifference > 50 || columnsChanged) {
                    if (!isResizing) {
                        console.log(`[MasonryFix] Resize detectado: ${widthDifference}px de diferença`);
                        isResizing = true;
                    }
                    
                    clearTimeout(resizeTimer);
                    
                    resizeTimer = setTimeout(() => {
                        console.log('[MasonryFix] Executando relayout após resize...');
                        previousWidth = currentWidth;
                        previousColumns = this.columns;
                        isResizing = false;
                        
                        this.relayout().catch(error => {
                            console.error('[MasonryFix] Erro durante relayout:', error);
                        });
                    }, 300);
                }
            };
            
            // Event listeners
            window.addEventListener('resize', handleResize, { passive: true });
            
            // Orientação mobile
            window.addEventListener('orientationchange', () => {
                console.log('[MasonryFix] Mudança de orientação detectada');
                setTimeout(handleResize, 100);
            });
            
            // Detecta mudanças de fonte/zoom
            if ('ResizeObserver' in window) {
                const resizeObserver = new ResizeObserver(() => {
                    handleResize();
                });
                resizeObserver.observe(this.container);
            }
        };
    }

    // Funções globais para debug e controle
    window.forceMasonryRelayout = function() {
        if (window.masonry && window.masonry.relayout) {
            console.log('[MasonryFix] Forçando reorganização manual...');
            window.masonry.updateColumns();
            return window.masonry.relayout();
        } else {
            console.error('[MasonryFix] Instância Masonry não encontrada');
        }
    };

    window.debugMasonry = function() {
        console.log('╔══════════════════════════════════╗');
        console.log('║      DEBUG MASONRY AVANÇADO      ║');
        console.log('╚══════════════════════════════════╝');
        
        if (!window.masonry) {
            console.error('❌ Masonry não inicializado!');
            console.log('Dica: Verifique se o feeds.js foi carregado corretamente');
            return;
        }
        
        const container = window.masonry.container;
        const items = container.querySelectorAll('.result');
        
        console.log('📊 Informações Gerais:');
        console.log('   Container:', container);
        console.log('   Largura:', container.offsetWidth + 'px');
        console.log('   Altura:', container.style.height || 'auto');
        console.log('   Colunas:', window.masonry.columns);
        console.log('   Gap:', window.masonry.gap + 'px');
        console.log('   Items:', items.length);
        
        console.log('\n📏 Alturas das Colunas:');
        if (window.masonry.columnHeights && window.masonry.columnHeights.length > 0) {
            window.masonry.columnHeights.forEach((height, index) => {
                console.log(`   Coluna ${index}: ${height}px`);
            });
        } else {
            console.log('   Não inicializadas');
        }
        
        // Análise de items
        console.log('\n🔍 Análise de Items:');
        let problemCount = 0;
        let hiddenCount = 0;
        let mispositionedCount = 0;
        
        items.forEach((item, index) => {
            const rect = item.getBoundingClientRect();
            const styles = window.getComputedStyle(item);
            const problems = [];
            
            // Verifica problemas
            if (!item.style.left || item.style.left === 'auto') {
                problems.push('sem left');
                mispositionedCount++;
            }
            if (!item.style.top || item.style.top === 'auto') {
                problems.push('sem top');
                mispositionedCount++;
            }
            if (styles.position !== 'absolute' && window.innerWidth > 500) {
                problems.push('position não é absolute');
            }
            if (rect.width === 0 || rect.height === 0) {
                problems.push('dimensões zero');
                hiddenCount++;
            }
            if (styles.opacity === '0' || styles.visibility === 'hidden') {
                problems.push('invisível');
                hiddenCount++;
            }
            
            if (problems.length > 0) {
                problemCount++;
                console.warn(`   Item ${index}:`, problems.join(', '));
            }
        });
        
        // Resumo
        console.log('\n📈 Resumo:');
        if (problemCount === 0) {
            console.log('   ✅ Todos os items estão OK!');
        } else {
            console.error(`   ⚠️ ${problemCount} items com problemas`);
            if (hiddenCount > 0) console.error(`   👻 ${hiddenCount} items invisíveis`);
            if (mispositionedCount > 0) console.error(`   📍 ${mispositionedCount} items mal posicionados`);
            console.log('\n💡 Execute forceMasonryRelayout() para tentar corrigir');
        }
        
        // Performance
        const performanceData = performance.getEntriesByType('measure').filter(entry => 
            entry.name.includes('masonry') || entry.name.includes('relayout')
        );
        
        if (performanceData.length > 0) {
            console.log('\n⚡ Performance:');
            performanceData.slice(-5).forEach(entry => {
                console.log(`   ${entry.name}: ${entry.duration.toFixed(2)}ms`);
            });
        }
        
        console.log('\n🛠️ Comandos Disponíveis:');
        console.log('   forceMasonryRelayout() - Força reorganização');
        console.log('   testMasonryPerformance() - Testa performance');
        console.log('   fixMasonryProblems() - Tenta corrigir problemas automaticamente');
        console.log('╚══════════════════════════════════╝');
    };

    // Função para testar performance
    window.testMasonryPerformance = async function() {
        if (!window.masonry) {
            console.error('Masonry não inicializado');
            return;
        }
        
        console.log('🏃 Iniciando teste de performance...');
        
        const iterations = 5;
        const times = [];
        
        for (let i = 0; i < iterations; i++) {
            const start = performance.now();
            await window.masonry.relayout();
            const end = performance.now();
            times.push(end - start);
            console.log(`   Iteração ${i + 1}: ${(end - start).toFixed(2)}ms`);
        }
        
        const average = times.reduce((a, b) => a + b, 0) / times.length;
        console.log(`\n📊 Média: ${average.toFixed(2)}ms`);
        console.log(`📈 Mínimo: ${Math.min(...times).toFixed(2)}ms`);
        console.log(`📉 Máximo: ${Math.max(...times).toFixed(2)}ms`);
    };

    // Função para corrigir problemas automaticamente
    window.fixMasonryProblems = function() {
        if (!window.masonry) {
            console.error('Masonry não inicializado');
            return;
        }
        
        console.log('🔧 Tentando corrigir problemas...');
        
        const container = window.masonry.container;
        const items = container.querySelectorAll('.result');
        let fixed = 0;
        
        items.forEach((item, index) => {
            let needsFix = false;
            
            // Força visibilidade
            if (item.style.opacity === '0' || item.style.visibility === 'hidden') {
                item.style.opacity = '1';
                item.style.visibility = 'visible';
                needsFix = true;
            }
            
            // Remove classes problemáticas
            item.classList.remove('loading', 'error');
            
            // Garante que tem classe show
            if (!item.classList.contains('show')) {
                item.classList.add('show');
                needsFix = true;
            }
            
            if (needsFix) {
                fixed++;
            }
        });
        
        if (fixed > 0) {
            console.log(`✅ ${fixed} items corrigidos`);
            console.log('🔄 Executando relayout...');
            forceMasonryRelayout();
        } else {
            console.log('✅ Nenhum problema óbvio encontrado');
        }
    };

    // Auto-correção ao carregar
    document.addEventListener('DOMContentLoaded', function() {
        console.log('[MasonryFix] DOM carregado, iniciando verificações...');
        
        let checkCount = 0;
        const maxChecks = 50; // 5 segundos no máximo
        
        const checkMasonry = setInterval(() => {
            checkCount++;
            
            if (window.masonry) {
                clearInterval(checkMasonry);
                console.log('[MasonryFix] Masonry detectado, aguardando items...');
                
                // Aguarda items serem carregados
                setTimeout(() => {
                    const items = document.querySelectorAll('.result');
                    if (items.length > 0) {
                        console.log(`[MasonryFix] ${items.length} items encontrados`);
                        
                        // Verifica se precisa correção
                        let needsFix = false;
                        items.forEach(item => {
                            if (!item.style.left || item.style.left === 'auto' || 
                                !item.style.top || item.style.top === 'auto') {
                                needsFix = true;
                            }
                        });
                        
                        if (needsFix) {
                            console.log('[MasonryFix] Problemas detectados, aplicando correção automática...');
                            forceMasonryRelayout();
                        } else {
                            console.log('[MasonryFix] Layout OK, nenhuma correção necessária');
                        }
                    }
                }, 1000);
            } else if (checkCount >= maxChecks) {
                clearInterval(checkMasonry);
                console.warn('[MasonryFix] Timeout - Masonry não foi inicializado');
            }
        }, 100);
    });

    // MutationObserver para detectar novos items
    if (window.MutationObserver) {
        let observer = null;
        let debounceTimer = null;
        
        const startObserver = () => {
            const container = document.querySelector('.result-list');
            if (!container) {
                setTimeout(startObserver, 500);
                return;
            }
            
            if (observer) observer.disconnect();
            
            observer = new MutationObserver((mutations) => {
                let hasNewItems = false;
                
                mutations.forEach(mutation => {
                    mutation.addedNodes.forEach(node => {
                        if (node.nodeType === 1 && node.classList && node.classList.contains('result')) {
                            hasNewItems = true;
                        }
                    });
                });
                
                if (hasNewItems && window.masonry) {
                    clearTimeout(debounceTimer);
                    debounceTimer = setTimeout(() => {
                        console.log('[MasonryFix] Novos items detectados');
                        if (window.masonry.columns > 1) {
                            window.masonry.relayout();
                        }
                    }, 200);
                }
            });
            
            observer.observe(container, {
                childList: true,
                subtree: false
            });
            
            console.log('[MasonryFix] MutationObserver ativado');
        };
        
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', startObserver);
        } else {
            startObserver();
        }
    }

    // Monitora erros de imagem
    document.addEventListener('error', function(e) {
        if (e.target.tagName === 'IMG' && e.target.closest('.result')) {
            console.warn('[MasonryFix] Erro ao carregar imagem:', e.target.src);
            e.target.style.display = 'none';
            
            // Força relayout se necessário
            if (window.masonry && window.masonry.columns > 1) {
                setTimeout(() => {
                    window.masonry.relayout();
                }, 100);
            }
        }
    }, true);

    // Log final
    console.log('[MasonryFix] ✅ Sistema de correção carregado completamente');
    console.log('[MasonryFix] 💡 Use debugMasonry() para diagnósticos');
    console.log('[MasonryFix] 🔧 Use forceMasonryRelayout() para reorganizar');
    console.log('[MasonryFix] 🏃 Use testMasonryPerformance() para testar performance');
    console.log('[MasonryFix] 🛠️ Use fixMasonryProblems() para correções automáticas');

})();