<?php
require_once __DIR__ . '/bootstrap.php';

$action = route_action();

if ($action === 'get_pinfeeds') handle_get_pinfeeds();
if ($action === 'get_feed') handle_get_feed();
if ($action === 'get_trending') handle_get_trending();
if ($action === 'get_sources') handle_get_sources();
if ($action === 'get_hashtags') handle_get_hashtags();
if ($action === 'search') handle_search();

json_fail('UNKNOWN_ACTION', 'No such feeds action: ' . $action, 404);

function base_select_sql(): string {
    // Pre-aggregated view counts across windows
    return "
    SELECT 
        p.id, p.title, p.link, p.thumbnail, p.description, p.pubDate, p.author, p.source_domain, p.likes, p.dislikes,
        COALESCE(v.views_1d,0) as views_1d,
        COALESCE(v.views_7d,0) as views_7d,
        COALESCE(v.views_14d,0) as views_14d,
        (COALESCE(v.views_1d,0)*1.0 + COALESCE(v.views_7d,0)*0.7 + COALESCE(v.views_14d,0)*0.3 + COALESCE(p.likes,0)*0.5 - COALESCE(p.dislikes,0)*0.2) as trending_score
    FROM pinfeeds p
    LEFT JOIN (
        SELECT 
            feed_id,
            SUM(CASE WHEN viewed_at >= NOW() - INTERVAL 1 DAY THEN 1 ELSE 0 END) as views_1d,
            SUM(CASE WHEN viewed_at >= NOW() - INTERVAL 7 DAY THEN 1 ELSE 0 END) as views_7d,
            SUM(CASE WHEN viewed_at >= NOW() - INTERVAL 14 DAY THEN 1 ELSE 0 END) as views_14d
        FROM feed_views
        GROUP BY feed_id
    ) v ON v.feed_id = p.id
    ";
}

function build_filters(&$params): array {
    $filters = [];

    $q = trim((string)param('q', ''));
    if ($q !== '') {
        // Use FULLTEXT where possible, otherwise fallback to LIKE
        $q_clean = preg_replace('/[^\p{L}\p{N}\s\-\+\*\.]/u', ' ', $q);
        $terms = array_filter(explode(' ', $q_clean));
        if ($terms) {
            $bool = '+' . implode(' +', array_map(function($t){ 
                $t = trim($t);
                if (mb_strlen($t) < 2) return $t;
                return $t . '*';
            }, $terms));
            $filters[] = "(MATCH(p.title) AGAINST (:q IN BOOLEAN MODE) OR MATCH(p.description) AGAINST (:q IN BOOLEAN MODE) OR MATCH(p.tags) AGAINST (:q IN BOOLEAN MODE) OR p.title LIKE :q_like OR p.description LIKE :q_like OR p.tags LIKE :q_like)";
            $params[':q'] = $bool;
            $params[':q_like'] = like_param($q);
        }
    }

    $domain = trim((string)param('domain',''));
    if ($domain !== '') {
        $filters[] = "p.source_domain = :domain";
        $params[':domain'] = $domain;
    }

    $author = trim((string)param('author',''));
    if ($author !== '') {
        $filters[] = "LOWER(p.author) = LOWER(:author)";
        $params[':author'] = $author;
    }

    $category = (string)param('category','');
    if ($category !== '') {
        if (is_numeric($category)) {
            $filters[] = "p.category = :catid";
            $params[':catid'] = (int)$category;
        } else {
            $filters[] = "p.categories LIKE :catlike";
            $params[':catlike'] = like_param($category);
        }
    }

    $mcid = param('main_category_id', '');
    if ($mcid !== '' && is_numeric($mcid)) {
        $filters[] = "p.main_category_id = :mcid";
        $params[':mcid'] = (int)$mcid;
    }

    $tags = param('tags', '');
    if ($tags !== '') {
        $tagList = sanitize_tags($tags);
        $i = 0;
        foreach ($tagList as $t) {
            $i++;
            $filters[] = "p.tags LIKE :tag{$i}";
            $params[":tag{$i}"] = like_param($t);
        }
    }

    $from = trim((string)param('from_date',''));
    if ($from !== '') {
        $filters[] = "p.pubDate >= :fromd";
        $params[':fromd'] = $from;
    }
    $to = trim((string)param('to_date',''));
    if ($to !== '') {
        $filters[] = "p.pubDate <= :tod";
        $params[':tod'] = $to;
    }

    return $filters;
}

function handle_get_pinfeeds() {
    $pdo = pdo();
    $params = [];
    $filters = build_filters($params);
    $where = build_where($filters);

    $order = normalize_order((string)param('order', (string)param('sort','latest')), ['latest','popular','trending'], 'latest');
    $page = clamp(as_int(param('page', 1), 1), 1, 1000000);
    $per = clamp(as_int(param('per_page', param('limit', 24)), 24), 1, 50);
    $offset = ($page - 1) * $per;

    $orderBy = "p.pubDate DESC, p.id DESC";
    if ($order === 'popular') {
        $orderBy = "COALESCE(v.views_7d,0) DESC, COALESCE(p.likes,0) DESC, p.pubDate DESC";
    } elseif ($order === 'trending') {
        $orderBy = "trending_score DESC";
    }

    // Build count + data queries
    $countSql = "SELECT COUNT(*) FROM pinfeeds p WHERE $where";
    $sql = base_select_sql() . " WHERE $where ORDER BY $orderBy LIMIT :limit OFFSET :offset";

    // Count
    $stmt = $pdo->prepare($countSql);
    foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
    $stmt->execute();
    $total = (int)$stmt->fetchColumn();

    // Data
    $stmt = $pdo->prepare($sql);
    foreach ($params as $k=>$v) $stmt->bindValue($k, $v);
    $stmt->bindValue(':limit', $per, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $rows = $stmt->fetchAll();

    $items = array_map('format_feed_row', $rows);

    json_ok([
        'meta' => [
            'page' => $page,
            'per_page' => $per,
            'total' => $total,
            'has_next' => ($offset + $per) < $total,
            'order' => $order,
        ],
        'items' => $items,
    ]);
}

function handle_get_feed() {
    $pdo = pdo();
    $id = as_int(param('id', 0), 0);
    if ($id <= 0) json_fail('BAD_REQUEST', 'Missing id', 422);

    $sql = base_select_sql() . " WHERE p.id = :id LIMIT 1";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->execute();
    $row = $stmt->fetch();
    if (!$row) json_fail('NOT_FOUND', 'Feed not found', 404);
    json_ok(['item' => format_feed_row($row)]);
}

function handle_get_trending() {
    $_GET['order'] = 'trending';
    handle_get_pinfeeds();
}

function handle_get_sources() {
    $pdo = pdo();
    $sql = "SELECT source_domain, COUNT(*) as count FROM pinfeeds GROUP BY source_domain ORDER BY count DESC LIMIT 500";
    $rows = $pdo->query($sql)->fetchAll();
    json_ok(['sources' => $rows]);
}

function handle_get_hashtags() {
    $pdo = pdo();
    $tags = [];
    if (table_exists('hashtags')) {
        $tags = $pdo->query("SELECT tag, count FROM hashtags ORDER BY count DESC, tag ASC LIMIT 500")->fetchAll();
    } else {
        // derive from pinfeeds.tags
        $rows = $pdo->query("SELECT tags FROM pinfeeds WHERE tags IS NOT NULL AND tags <> '' LIMIT 20000")->fetchAll(PDO::FETCH_COLUMN);
        $map = [];
        foreach ($rows as $t) {
            foreach (preg_split('/[,;|]/', $t) as $tok) {
                $tok = trim($tok);
                if ($tok==='') continue;
                $k = mb_strtolower($tok);
                $map[$k] = ($map[$k]??0)+1;
            }
        }
        arsort($map);
        $tags = [];
        foreach (array_slice($map, 0, 500, true) as $k=>$v) $tags[] = ['tag'=>$k,'count'=>$v];
    }
    json_ok(['hashtags' => $tags]);
}

function handle_search() {
    handle_get_pinfeeds();
}
