// init-fix.js - Corrige conflitos de inicialização entre múltiplos arquivos JS
// Adicione este arquivo ANTES de todos os outros scripts no index.php

(function() {
    'use strict';
    
    console.log('🔧 Iniciando correção de conflitos...');
    
    // ==========================================
    // SISTEMA DE INICIALIZAÇÃO SEQUENCIAL
    // ==========================================
    
    class InitializationManager {
        constructor() {
            this.modules = new Map();
            this.loadOrder = [];
            this.currentlyLoading = new Set();
            this.loaded = new Set();
            this.initQueue = [];
            this.isReady = false;
        }
        
        // Registra um módulo
        registerModule(name, dependencies = [], initFunction = null) {
            this.modules.set(name, {
                dependencies,
                initFunction,
                loaded: false
            });
            
            console.log(`📦 Módulo registrado: ${name}`);
        }
        
        // Marca um módulo como carregado
        markAsLoaded(name) {
            if (this.modules.has(name)) {
                this.modules.get(name).loaded = true;
                this.loaded.add(name);
                console.log(`✅ Módulo carregado: ${name}`);
                
                // Verifica se pode inicializar outros módulos
                this.checkDependencies();
            }
        }
        
        // Verifica dependências e inicializa módulos prontos
        checkDependencies() {
            for (const [name, module] of this.modules) {
                if (!module.loaded && this.areDependenciesLoaded(module.dependencies)) {
                    this.initializeModule(name, module);
                }
            }
            
            // Verifica se todos os módulos foram carregados
            if (this.loaded.size === this.modules.size && !this.isReady) {
                this.onAllModulesReady();
            }
        }
        
        // Verifica se todas as dependências foram carregadas
        areDependenciesLoaded(dependencies) {
            return dependencies.every(dep => this.loaded.has(dep));
        }
        
        // Inicializa um módulo específico
        initializeModule(name, module) {
            if (module.initFunction && typeof module.initFunction === 'function') {
                try {
                    console.log(`🚀 Inicializando: ${name}`);
                    module.initFunction();
                    module.loaded = true;
                    this.loaded.add(name);
                } catch (error) {
                    console.error(`❌ Erro ao inicializar ${name}:`, error);
                }
            } else {
                module.loaded = true;
                this.loaded.add(name);
            }
        }
        
        // Chamado quando todos os módulos estão prontos
        onAllModulesReady() {
            this.isReady = true;
            console.log('🎉 Todos os módulos carregados! Sistema pronto.');
            
            // Executa queue de inicialização
            this.initQueue.forEach(fn => {
                try {
                    fn();
                } catch (error) {
                    console.error('❌ Erro na queue de inicialização:', error);
                }
            });
            
            this.initQueue = [];
            
            // Dispara evento customizado
            window.dispatchEvent(new CustomEvent('systemReady'));
        }
        
        // Adiciona função à queue de inicialização
        onReady(fn) {
            if (this.isReady) {
                fn();
            } else {
                this.initQueue.push(fn);
            }
        }
    }
    
    // Instância global do gerenciador
    window.initManager = new InitializationManager();
    
    // ==========================================
    // PREVENÇÃO DE CONFLITOS
    // ==========================================
    
    // Previne múltiplas execuções do loadFeeds
    let loadFeedsExecuted = false;
    window.preventDuplicateLoadFeeds = function() {
        if (loadFeedsExecuted) {
            console.log('⚠️ loadFeeds já foi executado, prevenindo duplicação');
            return true;
        }
        loadFeedsExecuted = true;
        return false;
    };
    
    // Sistema de debounce para funções críticas
    window.debounce = function(func, wait, immediate) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                timeout = null;
                if (!immediate) func.apply(this, args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func.apply(this, args);
        };
    };
    
    // ==========================================
    // CORREÇÃO DE VARIÁVEIS GLOBAIS
    // ==========================================
    
    // Inicializa variáveis globais de forma segura
    function initializeGlobalVariables() {
        if (typeof window.feeds === 'undefined') {
            window.feeds = {};
            console.log('📝 Variável feeds inicializada');
        }
        
        if (typeof window.displayedFeeds === 'undefined') {
            window.displayedFeeds = {};
            console.log('📝 Variável displayedFeeds inicializada');
        }
        
        if (typeof window.page === 'undefined') {
            window.page = 1;
            console.log('📝 Variável page inicializada');
        }
        
        if (typeof window.loading === 'undefined') {
            window.loading = false;
            console.log('📝 Variável loading inicializada');
        }
        
        if (typeof window.masonry === 'undefined') {
            window.masonry = null;
            console.log('📝 Variável masonry inicializada');
        }
    }
    
    // ==========================================
    // CORREÇÃO DE AJAX/FETCH
    // ==========================================
    
    // Wrapper para requisições AJAX com retry automático
    window.safeAjax = function(options) {
        const defaultOptions = {
            timeout: 10000,
            retries: 2,
            retryDelay: 1000
        };
        
        const finalOptions = Object.assign({}, defaultOptions, options);
        
        function makeRequest(attempt = 0) {
            return new Promise((resolve, reject) => {
                const xhr = new XMLHttpRequest();
                
                xhr.timeout = finalOptions.timeout;
                xhr.open(finalOptions.type || 'GET', finalOptions.url, true);
                
                if (finalOptions.dataType === 'json') {
                    xhr.setRequestHeader('Accept', 'application/json');
                }
                
                xhr.onload = function() {
                    if (xhr.status >= 200 && xhr.status < 300) {
                        let response = xhr.responseText;
                        
                        if (finalOptions.dataType === 'json') {
                            try {
                                response = JSON.parse(response);
                            } catch (e) {
                                console.error('❌ Erro ao fazer parse JSON:', e);
                                reject(new Error('Invalid JSON response'));
                                return;
                            }
                        }
                        
                        resolve(response);
                    } else {
                        reject(new Error(`HTTP ${xhr.status}: ${xhr.statusText}`));
                    }
                };
                
                xhr.onerror = function() {
                    if (attempt < finalOptions.retries) {
                        console.log(`🔄 Tentativa ${attempt + 1}/${finalOptions.retries + 1} falhou, tentando novamente...`);
                        setTimeout(() => {
                            makeRequest(attempt + 1).then(resolve).catch(reject);
                        }, finalOptions.retryDelay);
                    } else {
                        reject(new Error('Network error'));
                    }
                };
                
                xhr.ontimeout = function() {
                    if (attempt < finalOptions.retries) {
                        console.log(`⏰ Timeout na tentativa ${attempt + 1}, tentando novamente...`);
                        setTimeout(() => {
                            makeRequest(attempt + 1).then(resolve).catch(reject);
                        }, finalOptions.retryDelay);
                    } else {
                        reject(new Error('Request timeout'));
                    }
                };
                
                // Envia dados se for POST
                if (finalOptions.data) {
                    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
                    xhr.send(new URLSearchParams(finalOptions.data).toString());
                } else {
                    xhr.send();
                }
            });
        }
        
        return makeRequest();
    };
    
    // ==========================================
    // REGISTRA MÓDULOS DO SISTEMA
    // ==========================================
    
    // Registra módulos na ordem correta
    initManager.registerModule('jquery');
    initManager.registerModule('utils', ['jquery']);
    initManager.registerModule('feeds', ['jquery', 'utils']);
    initManager.registerModule('masonry', ['jquery', 'feeds']);
    initManager.registerModule('search', ['jquery', 'feeds']);
    initManager.registerModule('ui', ['jquery']);
    initManager.registerModule('main', ['jquery', 'feeds', 'ui']);
    initManager.registerModule('loadFeeds', ['jquery', 'feeds', 'main'], function() {
        // Função de inicialização segura do loadFeeds
        if (typeof window.loadFeeds === 'function' && !window.preventDuplicateLoadFeeds()) {
            console.log('🚀 Executando loadFeeds inicial...');
            window.loadFeeds().then(() => {
                console.log('✅ loadFeeds inicial concluído');
            }).catch(error => {
                console.error('❌ Erro no loadFeeds inicial:', error);
            });
        }
    });
    
    // ==========================================
    // INICIALIZAÇÃO DO DOM
    // ==========================================
    
    function initializeSystem() {
        console.log('🚀 Inicializando sistema...');
        
        // Inicializa variáveis globais
        initializeGlobalVariables();
        
        // Marca jQuery como carregado se disponível
        if (typeof $ !== 'undefined') {
            initManager.markAsLoaded('jquery');
        }
        
        // Detecta quando outros scripts são carregados
        const originalLog = console.log;
        
        // Observer para detectar quando scripts específicos são carregados
        const checkForModules = setInterval(() => {
            // Verifica utils.js
            if (typeof window.extractDomainFromUrl === 'function' && !initManager.loaded.has('utils')) {
                initManager.markAsLoaded('utils');
            }
            
            // Verifica feeds.js
            if (typeof window.displayFeeds === 'function' && !initManager.loaded.has('feeds')) {
                initManager.markAsLoaded('feeds');
            }
            
            // Verifica masonry
            if (typeof window.MasonryLayout === 'function' && !initManager.loaded.has('masonry')) {
                initManager.markAsLoaded('masonry');
            }
            
            // Verifica search.js
            if (typeof window.searchController !== 'undefined' && !initManager.loaded.has('search')) {
                initManager.markAsLoaded('search');
            }
            
            // Verifica ui.js
            if (typeof window.showCustomNotification === 'function' && !initManager.loaded.has('ui')) {
                initManager.markAsLoaded('ui');
            }
            
            // Verifica main.js
            if (typeof window.topFunction === 'function' && !initManager.loaded.has('main')) {
                initManager.markAsLoaded('main');
            }
            
            // Verifica loadFeeds.js
            if (typeof window.loadFeeds === 'function' && !initManager.loaded.has('loadFeeds')) {
                initManager.markAsLoaded('loadFeeds');
            }
            
            // Para de verificar quando todos estiverem carregados
            if (initManager.isReady) {
                clearInterval(checkForModules);
            }
        }, 100);
        
        // Timeout de segurança
        setTimeout(() => {
            if (!initManager.isReady) {
                console.log('⏰ Timeout atingido, forçando inicialização...');
                clearInterval(checkForModules);
                initManager.onAllModulesReady();
            }
        }, 10000);
    }
    
    // ==========================================
    // INICIALIZAÇÃO
    // ==========================================
    
    // Inicializa quando DOM estiver pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initializeSystem);
    } else {
        initializeSystem();
    }
    
    // ==========================================
    // INTERFACE PÚBLICA
    // ==========================================
    
    // Função para outros scripts se registrarem como prontos
    window.moduleReady = function(moduleName) {
        initManager.markAsLoaded(moduleName);
    };
    
    // Função para executar código quando o sistema estiver pronto
    window.onSystemReady = function(callback) {
        initManager.onReady(callback);
    };
    
    // Debug functions
    window.debugInitialization = function() {
        console.log('=== DEBUG INICIALIZAÇÃO ===');
        console.log('Módulos registrados:', Array.from(initManager.modules.keys()));
        console.log('Módulos carregados:', Array.from(initManager.loaded));
        console.log('Sistema pronto:', initManager.isReady);
        console.log('Queue de inicialização:', initManager.initQueue.length);
        console.log('==========================');
    };
    
    console.log('✅ Sistema de correção de conflitos carregado');
})();