<?php
/**
 * ============================================
 * FLOWBOT DCI - NAVIGATION COMPONENT v3.0
 * ============================================
 * Professional navigation bar with context-aware links
 *
 * Features:
 * - Main navigation (Dashboard, New, History, Settings)
 * - Process-specific sub-navigation when viewing a process
 * - Real-time status indicator
 * - Mobile responsive
 */

declare(strict_types=1);

$baseUrl = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
$currentPage = $currentPage ?? 'dashboard';
$processId = $processId ?? null;

// Calculate process status if viewing a process
$processStatus = null;
$progressPercent = 0;
if ($processId && isset($data)) {
    $remainingLinks = 0;
    foreach ($data['phase_queues'] ?? [] as $queue) {
        $remainingLinks += count($queue);
    }
    $isComplete = ($remainingLinks === 0);
    $statusClass = $isComplete ? 'complete' : 'processing';
    if ($isComplete && ($data['error_links'] ?? 0) > 0) {
        $statusClass = 'has-errors';
    }

    // Progress percentage
    $total = max($data['total_links'] ?? 1, 1);
    $processed = ($data['processed_links'] ?? 0) + ($data['ignored_links'] ?? 0) + ($data['error_links'] ?? 0);
    $progressPercent = round(($processed / $total) * 100, 1);

    $processStatus = [
        'isComplete' => $isComplete,
        'statusClass' => $statusClass,
        'progressPercent' => $progressPercent,
        'processed' => $processed,
        'total' => $total
    ];
}
?>

<nav class="main-nav">
    <div class="nav-container">
        <!-- Brand -->
        <div class="nav-brand">
            <a href="<?= $baseUrl ?>/" class="brand-link">
                <span class="brand-logo">
                    <svg viewBox="0 0 32 32" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <rect width="32" height="32" rx="8" fill="url(#brand-gradient)"/>
                        <path d="M8 10h16M8 16h12M8 22h8" stroke="white" stroke-width="2.5" stroke-linecap="round"/>
                        <circle cx="24" cy="22" r="4" fill="#10B981"/>
                        <defs>
                            <linearGradient id="brand-gradient" x1="0" y1="0" x2="32" y2="32">
                                <stop stop-color="#3B82F6"/>
                                <stop offset="1" stop-color="#8B5CF6"/>
                            </linearGradient>
                        </defs>
                    </svg>
                </span>
                <span class="brand-text">Flowb0t DCI</span>
                <span class="brand-version">v3.0</span>
            </a>
        </div>

        <!-- Main Navigation -->
        <div class="nav-main">
            <a href="<?= $baseUrl ?>/"
               class="nav-link <?= $currentPage === 'dashboard' ? 'active' : '' ?>"
               title="Dashboard">
                <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="3" width="7" height="7" rx="1"/>
                    <rect x="14" y="3" width="7" height="7" rx="1"/>
                    <rect x="3" y="14" width="7" height="7" rx="1"/>
                    <rect x="14" y="14" width="7" height="7" rx="1"/>
                </svg>
                <span class="nav-text">Dashboard</span>
            </a>

            <a href="<?= $baseUrl ?>/new"
               class="nav-link <?= $currentPage === 'new' || $currentPage === 'form' ? 'active' : '' ?>"
               title="New Process">
                <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="9"/>
                    <line x1="12" y1="8" x2="12" y2="16"/>
                    <line x1="8" y1="12" x2="16" y2="12"/>
                </svg>
                <span class="nav-text">New Process</span>
            </a>

            <a href="<?= $baseUrl ?>/history"
               class="nav-link <?= $currentPage === 'history' ? 'active' : '' ?>"
               title="History">
                <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="9"/>
                    <polyline points="12,7 12,12 16,14"/>
                </svg>
                <span class="nav-text">History</span>
            </a>

            <a href="<?= $baseUrl ?>/crawler"
               class="nav-link <?= $currentPage === 'crawler' ? 'active' : '' ?>"
               title="Deep Crawler">
                <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="11" cy="11" r="8"/>
                    <line x1="21" y1="21" x2="16.65" y2="16.65"/>
                    <line x1="11" y1="8" x2="11" y2="14"/>
                    <line x1="8" y1="11" x2="14" y2="11"/>
                </svg>
                <span class="nav-text">Crawler</span>
            </a>

            <a href="<?= $baseUrl ?>/settings"
               class="nav-link <?= $currentPage === 'settings' ? 'active' : '' ?>"
               title="Settings">
                <svg class="nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="3"/>
                    <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/>
                </svg>
                <span class="nav-text">Settings</span>
            </a>
        </div>

        <!-- Right Side: Process Status or Quick Actions -->
        <div class="nav-right">
            <?php if ($processId && $processStatus): ?>
                <!-- Process Status Indicator -->
                <div class="nav-process-status">
                    <div class="process-mini-progress" title="<?= $processStatus['progressPercent'] ?>% complete">
                        <div class="progress-fill" style="width: <?= $processStatus['progressPercent'] ?>%"></div>
                    </div>
                    <span class="status-dot <?= $processStatus['statusClass'] ?>"></span>
                    <span class="process-id-short" title="Process: <?= htmlspecialchars($processId) ?>">
                        #<?= htmlspecialchars(substr($processId, 0, 8)) ?>
                    </span>
                </div>
            <?php else: ?>
                <!-- Quick Action Button -->
                <a href="<?= $baseUrl ?>/new" class="btn btn-primary btn-sm">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="16" height="16">
                        <line x1="12" y1="5" x2="12" y2="19"/>
                        <line x1="5" y1="12" x2="19" y2="12"/>
                    </svg>
                    New
                </a>
            <?php endif; ?>

            <!-- Mobile Menu Toggle -->
            <button class="nav-mobile-toggle" onclick="toggleMobileNav()" aria-label="Toggle menu">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="24" height="24">
                    <line x1="3" y1="6" x2="21" y2="6"/>
                    <line x1="3" y1="12" x2="21" y2="12"/>
                    <line x1="3" y1="18" x2="21" y2="18"/>
                </svg>
            </button>
        </div>
    </div>
</nav>

<?php if ($processId): ?>
<!-- Process Sub-Navigation -->
<nav class="sub-nav">
    <div class="sub-nav-container">
        <div class="sub-nav-links">
            <a href="?process_id=<?= urlencode($processId) ?>"
               class="sub-nav-link <?= $currentPage === 'progress' ? 'active' : '' ?>">
                <svg class="sub-nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M12 20v-6m0 0V4m0 10h6m-6 0H6"/>
                </svg>
                Progress
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=domains"
               class="sub-nav-link <?= $currentPage === 'domains' ? 'active' : '' ?>">
                <svg class="sub-nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <circle cx="12" cy="12" r="10"/>
                    <line x1="2" y1="12" x2="22" y2="12"/>
                    <path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/>
                </svg>
                Domains
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=logs"
               class="sub-nav-link <?= $currentPage === 'logs' ? 'active' : '' ?>">
                <svg class="sub-nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                    <polyline points="14 2 14 8 20 8"/>
                    <line x1="16" y1="13" x2="8" y2="13"/>
                    <line x1="16" y1="17" x2="8" y2="17"/>
                    <polyline points="10 9 9 9 8 9"/>
                </svg>
                Logs
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=export"
               class="sub-nav-link <?= $currentPage === 'export' ? 'active' : '' ?>">
                <svg class="sub-nav-icon" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/>
                    <polyline points="7 10 12 15 17 10"/>
                    <line x1="12" y1="15" x2="12" y2="3"/>
                </svg>
                Export
            </a>
        </div>

        <!-- Process Quick Stats -->
        <div class="sub-nav-stats">
            <span class="stat-item success" title="Successful">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                    <polyline points="20 6 9 17 4 12"/>
                </svg>
                <?= number_format($data['processed_links'] ?? 0) ?>
            </span>
            <span class="stat-item error" title="Errors">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                    <line x1="18" y1="6" x2="6" y2="18"/>
                    <line x1="6" y1="6" x2="18" y2="18"/>
                </svg>
                <?= number_format($data['error_links'] ?? 0) ?>
            </span>
            <span class="stat-item total" title="Total">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="14" height="14">
                    <line x1="8" y1="6" x2="21" y2="6"/>
                    <line x1="8" y1="12" x2="21" y2="12"/>
                    <line x1="8" y1="18" x2="21" y2="18"/>
                    <line x1="3" y1="6" x2="3.01" y2="6"/>
                    <line x1="3" y1="12" x2="3.01" y2="12"/>
                    <line x1="3" y1="18" x2="3.01" y2="18"/>
                </svg>
                <?= number_format($data['total_links'] ?? 0) ?>
            </span>
        </div>
    </div>
</nav>
<?php endif; ?>

<!-- Mobile Navigation Overlay -->
<div class="nav-mobile-overlay" id="mobileNavOverlay" onclick="closeMobileNav()"></div>
<div class="nav-mobile-menu" id="mobileNavMenu">
    <div class="mobile-menu-header">
        <span class="brand-text">Flowb0t DCI</span>
        <button class="mobile-close" onclick="closeMobileNav()" aria-label="Close menu">
            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="24" height="24">
                <line x1="18" y1="6" x2="6" y2="18"/>
                <line x1="6" y1="6" x2="18" y2="18"/>
            </svg>
        </button>
    </div>
    <div class="mobile-menu-links">
        <a href="<?= $baseUrl ?>/" class="mobile-link <?= $currentPage === 'dashboard' ? 'active' : '' ?>">
            Dashboard
        </a>
        <a href="<?= $baseUrl ?>/new" class="mobile-link <?= $currentPage === 'new' ? 'active' : '' ?>">
            New Process
        </a>
        <a href="<?= $baseUrl ?>/history" class="mobile-link <?= $currentPage === 'history' ? 'active' : '' ?>">
            History
        </a>
        <a href="<?= $baseUrl ?>/crawler" class="mobile-link <?= $currentPage === 'crawler' ? 'active' : '' ?>">
            Deep Crawler
        </a>
        <a href="<?= $baseUrl ?>/settings" class="mobile-link <?= $currentPage === 'settings' ? 'active' : '' ?>">
            Settings
        </a>
    </div>
    <?php if ($processId): ?>
    <div class="mobile-menu-section">
        <div class="mobile-section-title">Current Process</div>
        <a href="?process_id=<?= urlencode($processId) ?>" class="mobile-link">Progress</a>
        <a href="?process_id=<?= urlencode($processId) ?>&page=domains" class="mobile-link">Domains</a>
        <a href="?process_id=<?= urlencode($processId) ?>&page=logs" class="mobile-link">Logs</a>
        <a href="?process_id=<?= urlencode($processId) ?>&page=export" class="mobile-link">Export</a>
    </div>
    <?php endif; ?>
</div>

<script>
function toggleMobileNav() {
    document.getElementById('mobileNavOverlay').classList.toggle('active');
    document.getElementById('mobileNavMenu').classList.toggle('active');
    document.body.classList.toggle('nav-open');
}

function closeMobileNav() {
    document.getElementById('mobileNavOverlay').classList.remove('active');
    document.getElementById('mobileNavMenu').classList.remove('active');
    document.body.classList.remove('nav-open');
}

// Close on escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeMobileNav();
    }
});
</script>
